/*************************************************************************;
%** PROGRAM: assign_assessments_to_cmgs.sas      
%** PURPOSE: assign assessments to case mix groups     
%** DATE CREATED:       12/09/2016
%** DATE LAST MODIFIED: 11/12/2018         
%** 
%** RESTRICTED RIGHTS NOTICE (SEPT 2014)
%** (a) This computer software is submitted with restricted rights under Government Agreement No. CMS: DataLink 3 OY1. It may not be used, reproduced, or disclosed by the Government except as provided in paragraph (b) of this Notice or as otherwise expressly stated in the agreement. 
%** (b) This computer software may be:
%**    (1) Used or copied for use in or with the computer or computers for which it was acquired, including use at any Government installation to which such computer or computers may be transferred; 
%**    (2) Used or copied for use in a backup computer if any computer for which it was acquired is inoperative; 
%**    (3) Reproduced for safekeeping (archives) or backup purposes; 
%**    (4) Modified, adapted, or combined with other computer software, provided that the modified, combined, or adapted portions of the derivative software are made subject to the same restricted rights; 
%**    (5) Disclosed to and reproduced for use by support service Recipients in accordance with subparagraphs (b)(1) through (4) of this clause, provided the Government makes such disclosure or reproduction subject to these restricted rights; and
%**    (6) Used or copied for use in or transferred to a replacement computer. 
%** (c) Notwithstanding the foregoing, if this computer software is published copyrighted computer software, it is licensed to the Government, without disclosure prohibitions, with the minimum rights set forth in paragraph (b) of this clause. 
%** (d) Any other rights or limitations regarding the use, duplication, or disclosure of this computer software are to be expressly stated in, or incorporated in, the agreement. 
%** (e) This Notice shall be marked on any reproduction of this computer software, in whole or in part. 
%** (End of notice)   
%*************************************************************************/

   /**********************************************************************************************************************
   *                                                                                                                     *
   *  assign_assessments_to_cmgs                                                                                         * 
   *                                                                                                                     *
   *      Part 1: List Relevant Variables and Specify Formats                                                            *
   *          List out 5-day MDS variables that should be used to classify patients into case mix groups                 *
   *          Format function score, cognitive score, and NTA score into bins, and nursing RUGs into collasped version   *
   *          Store the list of SLP and NTA comorbidities into a macro variable to be used for looping later             *
   *      Part 2: Create Grouping Variables and Assign Assessments to Case Mix Groups                                    *
   *          Part 2.1: Read in intermediate datasets for mapping purposes                                               *
   *          Part 2.2: Create grouping variables for each payment component                                             *
   *          Part 2.3: Assign assessments to case mix groups using constructed grouping variables                       *
   *          Part 2.4: Assign HIPPS code to assessments using case mix groups                                           *
   *                                                                                                                     *
   **********************************************************************************************************************/

   %macro assign_assessments_to_cmgs(
      
                  code_folder_path                 = ,
                  input_dataset_folder_path        = ,
                  intermediate_mapping_folder_path = ,
                  output_dataset_folder_path       = ,   
      
                  in_assessments_dsn   = , 
                  out_assessments_dsn  = ,
      ); 
      
      %* set system option;
      options ls=220 ps=max nocenter mprint noxwait compress=yes fullstimer mergenoby=warn extendobscounter=no sasautos = (sasautos, "&code_folder_path.");

      %* define libname shortcuts; 
      libname in    "&input_dataset_folder_path.";
      libname inter "&intermediate_mapping_folder_path."; 
      libname out   "&output_dataset_folder_path.";
         
      ************************************************************************************************************************************;
      *                                                                                                                                   ;
      * Part 1: List Relevant Variables and Specify Final Formats                                                                         ;
      *                                                                                                                                   ;
      ************************************************************************************************************************************; 

      %************************************************************************************************************************;
      %* List out 5-day MDS variables that will be used to classify patients into case mix groups;  
      %* (1)  items for calculating function score;
      %* (2)  items for calculating cogntive status score;
      %* (3)  items for constructing SLP case mix groups;
      %* (4)  items for calculating NTA comorbidity score - HIV condition is removed from the list since it will be handled by Pricer;
      %* (5)  items for calculating NTA comorbidity score that can be skipped and therefore have many missing values due to skip patterns in MDS;
      %* (6)  items for constructing Nursing case mix groups;
      %* (7)  items for constructing Nursing case mix groups that can be skipped and therefore have many missing values due to skip patterns in MDS;
      %* (8)  items i8000 records up to 10 diagnoses using ICD10 codes;
      %* (9)  item  i0020b records the primary SNF diagnosis;         
      %* (10) items j2000 records surgical information;
      %************************************************************************************************************************; 

      %let mds_function_item_list         = gg0130a1_eatg_self_admsn_cd  gg0130b1_oral_hygne_admsn_cd  gg0130c1_toilt_hygne_admsn_cd 
                                            gg0170b1_sit_admsn_cd        gg0170c1_lyng_admsn_cd        gg0170d1_stand_admsn_cd         
                                            gg0170e1_chr_trnsf_admsn_cd  gg0170f1_toilt_trnsf_admsn_cd gg0170i1_wlk_10_admsn_cd     
                                            gg0170j1_wlk_50_admsn_cd     gg0170k1_wlk_150_admsn_cd;       

      %let mds_cognitive_item_list        = c1000_dcsn_mkng_cd           b0700_self_undrstod_cd        c0700_shrt_term_memry_cd 
                                            b0100_cmts_cd                bims_score;

      %let mds_slp_item_list              = i4300_aphasia_cd             i4500_strk_cd                 i4900_hemiplg_cd 
                                            i5500_brn_injury_cd          o0100e2_trchostmy_post_cd     o0100f2_vntltr_post_cd
                                            k0100z_no_swlwg_cd           k0510c2_altr_food_post_cd;

      %let mds_nta_item_list              = i2900_dm_cd                  k0510a2_pen_post_cd            i1300_ulcrtv_clts_cd
                                            o0100h2_iv_mdctn_post_cd     o0100f2_vntltr_post_cd         o0100i2_trnsfsn_post_cd
                                            i5200_ms_cd                  i6200_asthma_cd                i2500_wnd_infctn_cd        
                                            m1040a_ft_infctn_cd          m1040b_dbtc_ft_ulcr_cd         m1040c_othr_lsn_ft_cd  
                                            o0100e2_trchostmy_post_cd    i1700_mdro_cd                  o0100m2_isltn_post_cd
                                            o0100b2_rdtn_post_cd         h0100d_intrmtnt_cthtr_cd       o0100d2_sctng_post_cd        
                                            k0510b2_feedg_tube_post_cd   i5600_malntrtn_cd              h0100c_ostmy_cd;                                           
      %let mds_nta_optional_item_list     = k0710a2_cal_prntrl_post_cd   k0710b2_iv_tube_daily_post_cd  m0300d1_stg_4_ulcr_num;

      %let mds_nursing_item_list          = o0100a2_chmthrpy_post_cd     i2000_pneumo_cd             j1550b_vmtg_cd                k0300_wt_loss_cd              i2100_sptcmia_cd    
                                            i5100_quadplg_cd             j1100c_sob_lyg_cd           j1550a_fvr_cd                 e0900_wndr_cd                 m1200b_prsr_rdc_bed_cd
                                            m1200a_prsr_rdc_chr_cd       m1200c_trng_pgm_cd          m1200d_hydrtn_cd              m1200e_ulcr_care_cd           m1200g_aplctn_drsng_cd
                                            m1200h_aplctn_ontmnt_cd      i4400_crbrl_plsy_cd         i5300_prknsn_cd               i6300_rsprtry_failr_cd        o0100c2_oxgn_post_cd
                                            m1030_artrl_ulcr_num         o0100j2_dlys_post_cd        m1200f_srgcl_wnd_care_cd      m1040e_srgcl_wnd_cd           m1200i_aplctn_drsng_foot_cd
                                            m1040d_open_lsn_cd           m1040f_brn_cd               o0100a2_chmthrpy_post_cd      e0100b_dlsn_cd                e0100a_hllcntn_cd                                                           
                                            h0500_bwl_toiltg_pgm_cd      o0500f_wlkg_trng_num        o0500d_bed_mblty_trng_num     o0500a_psv_rom_num            o0500b_actv_rom_num
                                            o0500c_brc_astnc_num         o0500e_trnsfr_trng_num      o0500g_drsg_trng_num          o0500h_eatg_trng_num          o0500i_amputtn_trng_num      
                                            o0500j_commun_trng_num       e0200a_phys_bhvrl_cd        e0200b_vrbl_bhvrl_cd          e0200c_othr_bhvrl_cd          e0800_rjct_evaltn_cd
                                            k0510b2_feedg_tube_post_cd;
      %let mds_nursing_optional_item_list = d0300_mood_scre_num            d0600_stf_mood_scre_num    k0510b1_feedg_tube_prior_cd  k0510a1_pen_prior_cd          k0710a3_cal_prntrl_7_day_cd        
                                            k0710b3_iv_tube_daily_7_day_cd m0300b1_stg_2_ulcr_num     m0300c1_stg_3_ulcr_num       m0300f1_unstgbl_ulcr_esc_num  h0200c_crnt_toiltg_pgm_cd       
                                            n0350a_insln_injct_day_num     n0350b_insln_ordr_day_num  o0400d2_rt_day_num;                     

      %let mds_i8000_item_list            = i8000a_icd_1_cd i8000b_icd_2_cd i8000c_icd_3_cd i8000d_icd_4_cd i8000e_icd_5_cd i8000f_icd_6_cd i8000g_icd_7_cd i8000h_icd_8_cd i8000i_icd_9_cd i8000j_icd_10_cd;

      %let mds_i0020_item_list            = i0020b;

      %let mds_j2000_item_list            = J2300 J2310 J2320 J2330 J2400 J2410 J2420 J2499 J2500 J2510 J2520 J2530 J2599 J2600 J2610 J2620 J2699 J2700 J2710 J2799 J2800 J2810 J2899 J2900 J2910 J2920 J2930 J2940;

      %*******************************************************************************************************;         
      %* Group function score, cognitive score, NTA Score into bins and Collapse Nursing RUGs               ;
      %*******************************************************************************************************;         

      proc format;     
         %* Format PT and OT Final Group Variable - Function Score bin;
         value function_score_format
            0  - 5    = "1. 0-5"
            5  <-9    = "2. 6-9"
            9  <-23   = "3. 10-23"
            23 <-high = "4. 24"
            other     = "99. Unknown"  
         ; 

         %* Format SLP Final Group Variable - Cognitive Status bin;
         value $cognitive_format
            "1. Cognitively Intact"                                                     = "1. Cognitively Intact"
            "2. Mildly Impaired", "3. Moderately Impaired", "4. Severely Impaired"      = "2. Cognitively Impaired"
            other                                                                       = "99. Unknown"  
         ; 

         %* Format NTA Final Group Variable - Comorbidity Score Bin;
         value nta_score_format
            0         = "1. 0"   
            1  - 2    = "2. 1-2" 
            3  - 5    = "3. 3-5"
            6  - 8    = "4. 6-8"
            9  - 11   = "5. 9-11"
            12 - high = "6. 12+"  
            other     = "99. Unknown"
         ;
      run; 

      %* Format Nursing Final Group Variable - Collapsed RUGs;
      %let collapsed_rugs = 01 ES3 # 02 ES2 # 03 ES1 # 04 HE2 HD2 # 05 HE1 HD1 # 06 HC2 HB2 # 07 HC1 HB1 # 08 LE2 LD2 # 09 LE1 LD1 # 10 LC2 LB2 
                            # 11 LC1 LB1 # 12 CE2 CD2 # 13 CE1 CD1 # 14 CC2 CB2 # 15 CA2 # 16 CC1 CB1 # 17 CA1 # 18 BB2 BA2 # 19 BB1 BA1 # 20 PE2 PD2 # 21 PE1 PD1 
                            # 22 PC2 PB2 # 23 PA2 # 24 PC1 PB1 # 25 PA1;  

      %ref_loop(collapsed_rugs, &collapsed_rugs., dlm=#); 

      %* Transform function score variables into a series of index based variables; 
      %ref_loop(pt_ot_gg_items, GG0130A1_EATG_SELF_ADMSN_CD # GG0130B1_ORAL_HYGNE_ADMSN_CD # GG0130C1_TOILT_HYGNE_ADMSN_CD # GG0170B1_SIT_ADMSN_CD 
                               # GG0170C1_LYNG_ADMSN_CD # GG0170D1_STAND_ADMSN_CD # GG0170E1_CHR_TRNSF_ADMSN_CD # GG0170F1_TOILT_TRNSF_ADMSN_CD
                               # gg0170j1_wlk_50_admsn_cd # gg0170k1_wlk_150_admsn_cd, dlm =#);                                  
      %ref_loop(pt_ot_gg_items_label, EATG ORAL_HYGNE TOILT_HYGNE SIT LYNG STAND CHR_TRNSF TOILT_TRNSF wlk_50 wlk_150);       
      %ref_loop(nursing_gg_items, GG0130A1_EATG_SELF_ADMSN_CD # GG0130C1_TOILT_HYGNE_ADMSN_CD # GG0170B1_SIT_ADMSN_CD # GG0170C1_LYNG_ADMSN_CD #  
                                  GG0170D1_STAND_ADMSN_CD # GG0170E1_CHR_TRNSF_ADMSN_CD # GG0170F1_TOILT_TRNSF_ADMSN_CD, dlm =#);                                    
      %ref_loop(nursing_gg_items_label, EATG TOILT_HYGNE SIT LYNG STAND CHR_TRNSF TOILT_TRNSF);

      %* Transform nursing rug variables (behavior symptoms and restorative nursing) into a series of index based variables; 
      %ref_loop(behavior_symp, e0200a_phys_bhvrl_cd e0200b_vrbl_bhvrl_cd e0200c_othr_bhvrl_cd e0800_rjct_evaltn_cd e0900_wndr_cd);  
      %ref_loop(restorative_nursing, _nursing_mobility _nursing_rom o0500c_brc_astnc_num o0500e_trnsfr_trng_num 
                                 o0500g_drsg_trng_num o0500h_eatg_trng_num o0500i_amputtn_trng_num o0500j_commun_trng_num); 

      %*******************************************************************************************************;   
      %* Store lists of SLP and NTA comorbidities into macro variables to be used for looping later; 
      %*******************************************************************************************************;         

      %* store i8000 based SLP comorbidities into a macro variable;
      proc sql noprint;
         select distinct(condition) 
         into:  i8000_slp_cmrbd_list separated by " " 
         from inter.icd10_to_slp_cmrbd_mapping;
      run;

      %* store i8000 based NTA comorbidities into a macro variable;
      proc sql noprint;
         select distinct(condition) 
         into:  i8000_nta_cmrbd_list separated by " " 
         from inter.icd10_to_nta_cmrbd_mapping;
      run;  

      %* store all NTA comorbidities into a macro variable;
      proc sql noprint;
         select nta_cmrbd
         into : nta_cmrbd_list separated by ' '
         from inter.nta_cmrbd_list;
      run;         

      %* transform MDS I8000 active diagnosis variables into a series of index based variables; 
      %ref_loop(mds_i8000s, &mds_i8000_item_list.);
      %let condition_list  = &i8000_slp_cmrbd_list.  &i8000_nta_cmrbd_list.;
      %ref_loop(nta_cmrbds, &nta_cmrbd_list.);
      %ref_loop(all_conditions, &condition_list.);
      %let map_hash_name_list  = icd10_to_slp_cmrbd_map  icd10_to_nta_cmrbd_map;
      %ref_loop(map_hash_names, &map_hash_name_list.);

      ************************************************************************************************************************************;
      *                                                                                                                                   ;         
      * Part 2: Create Grouping Variables and Assign Assessments to Case Mix Groups                                                             ; 
      *                                                                                                                                   ;         
      ************************************************************************************************************************************;    

      data out.&out_assessments_dsn. (drop=_: icd_10_cm_code clinical_category_pt_ot clinical_category_slp condition nta_cmrbd nta_point cmg_component cmg_long cmg_short hipps_character);

         ************************************************************************************************************************************;
         *                                                                                                                                   ;            
         * Part 2.1: Read in Intermediate Datasets for Mapping Purposes                                                                      ;
         *                                                                                                                                   ;            
         ************************************************************************************************************************************; 

         if _N_=0 then set inter.icd10_to_clin_cat_mapping
                           inter.icd10_to_slp_cmrbd_mapping
                           inter.icd10_to_nta_cmrbd_mapping
                           inter.nta_cmrbd_list
                           inter.cmg_to_hipps_code_mapping;
         if _N_=1 then do;
            %*hash in ICD_10_CM code to clinical category mapping; 
            declare hash icd10_to_clin_cat_map(dataset: "inter.icd10_to_clin_cat_mapping");
            icd10_to_clin_cat_map.defineKey("icd_10_cm_code");
            icd10_to_clin_cat_map.defineData(all:"yes");
            icd10_to_clin_cat_map.defineDone(); 

            %*hash in ICD_10_CM code to SLP comorbidity mapping; 
            declare hash icd10_to_slp_cmrbd_map(dataset: "inter.icd10_to_slp_cmrbd_mapping", multidata:"yes");
            icd10_to_slp_cmrbd_map.defineKey("icd_10_cm_code");
            icd10_to_slp_cmrbd_map.defineData(all:"yes");
            icd10_to_slp_cmrbd_map.defineDone();  

            %*hash in ICD_10_CM code to NTA comorbidity mapping; 
            declare hash icd10_to_nta_cmrbd_map(dataset: "inter.icd10_to_nta_cmrbd_mapping", multidata:"yes");
            icd10_to_nta_cmrbd_map.defineKey("icd_10_cm_code");
            icd10_to_nta_cmrbd_map.defineData(all:"yes");
            icd10_to_nta_cmrbd_map.defineDone(); 

            %*hash in the list of NTA comorbidities and their corresponding points; 
            declare hash nta_cmrbd_point(dataset: "inter.nta_cmrbd_list");
            nta_cmrbd_point.defineKey("nta_cmrbd");
            nta_cmrbd_point.defineData(all:"yes");
            nta_cmrbd_point.defineDone();

            %*hash in Case Mix Group to HIPPS code mapping; 
            declare hash cmg_to_hipps(dataset: "inter.cmg_to_hipps_code_mapping");
            cmg_to_hipps.defineKey("cmg_long");
            cmg_to_hipps.defineData(all:"yes");
            cmg_to_hipps.defineDone();               
         end;      
         set in.&in_assessments_dsn. (keep = id assessment_indicator 
                                    &mds_function_item_list. &mds_cognitive_item_list. &mds_slp_item_list. &mds_nta_item_list. &mds_nta_optional_item_list. &mds_nursing_item_list. &mds_nursing_optional_item_list. &mds_i8000_item_list. &mds_i0020_item_list. &mds_j2000_item_list.);
         by id;

         ************************************************************************************************************************************;
         *                                                                                                                                   ;            
         * Part 2.2: Create Grouping Variables                                                                                               ;
         *                                                                                                                                   ;            
         ************************************************************************************************************************************;                         

         %*******************************************************************************************************;
         %****   Construct Clinical Categories;
         %****      maps each assessment to a clinical category based on the icd-10-cm code recorded on MDS item I0020B;
         %****      checks if an assessments should be reassigned to a surgical category based on MDS item J2000;
         %****      creates 3 intermediate grouping variables - default_clinical_category  resident_had_a_major_procedure  clinical_categories
         %****      creates 2 final grouping variables        - pt_ot_clinical_category slp_clinical_category;
         %*******************************************************************************************************;

         %* (1) Determine the clinical category of a assessment using ICD-10-CM codes recorded in MDS I0020B; 
         %* .find() searches the icd10_to_clin_cat_map hash table by the key, "i0020b" from the in_assessments_dsn dataset; 
         %* If the key is found, then it merges the clinical category information from the hash table; 
         if icd10_to_clin_cat_map.find(Key:i0020b)=0 then do; 
            clinical_categories     = default_clinical_category;
            pt_ot_clinical_category = clinical_category_pt_ot;
            slp_clinical_category   = clinical_category_slp;
         end;

         %* (2) Check the J2000 item list to see if an assessments should be reassigned to a surgical clinical category;
         %* check for Non-Orthopedic Surgery first;
         %* then Orthopedic Surgery (Except Major Joint Replacement or Spinal Surgery);
         %* lastly, Major Joint Replacement or Spinal Surgery;
         %* we follow this hierarchy, because the final clinical category assignment should pick the most expensive surgical category among the ones a assessment can possibly be assigned to;

         if resident_had_a_major_procedure = "May be Eligible for the Non-Orthopedic Surgery Category" then do; 
            %* if any J2000 items related to Non-Orthopedic Surgery is checked off;
            %* then reassign the assessment to the corresponding clinical category;            
            array non_ortho [*] J2600 J2610 J2620 J2700 J2710 J2800 J2810 J2900 J2910 J2920 J2930 J2940; 
            do _m=1 to dim(non_ortho);
               if non_ortho[_m] = "1" then do;
                  clinical_categories     = "Non-Orthopedic Surgery";
                  pt_ot_clinical_category = "4. Non-Orthopedic Surgery and Acute Neurologic";
                  slp_clinical_category   = "1. Non-Neurologic";
               end;
            end;               
         end;             

         if resident_had_a_major_procedure = "May be Eligible for One of the Two Orthopedic Surgery Categories" then do;      
            %* if any J2000 items related to Orthopedic Surgery (Except Major Joint Replacement or Spinal Surgery) is checked off;
            %* then reassign the assessment to the corresponding clinical category;
            array ortho_surg [*] J2500 J2510 J2520 J2530; 
            do _m=1 to dim(ortho_surg);
               if ortho_surg[_m] = "1" then do;
                  clinical_categories     = "Orthopedic Surgery (Except Major Joint Replacement or Spinal Surgery)";
                  pt_ot_clinical_category = "2. Other Orthopedic";
                  slp_clinical_category   = "1. Non-Neurologic";
               end;
            end; 
            %* if any J2000 items related to Major Joint Replacement or Spinal Surgery is checked off;
            %* then reassign the assessment to the corresponding clinical category;
            array major_joint [*] J2300 J2310 J2320 J2330 J2400 J2410 J2420; 
            do _m=1 to dim(major_joint);
               if major_joint[_m] = "1" then do;
                  clinical_categories     = "Major Joint Replacement or Spinal Surgery";
                  pt_ot_clinical_category = "1. Major Joint Replacement or Spinal Surgery";
                  slp_clinical_category   = "1. Non-Neurologic";
               end;
            end;                             
         end;            

         %*******************************************************************************************************;
         %****   Construct Function Scores (Section GG);
         %****      calls create_function_score macro;
         %****      creates 1 intermediate grouping variables - function_score;
         %****      creates 2 final grouping variables        - pt_ot_function_score nursing_function_score;
         %*******************************************************************************************************;

         %create_function_score();

         %*******************************************************************************************************;
         %***    Construct PDPM Cognitive Level;
         %***       creates 1 final grouping variable         - cognitive_level; 
         %*******************************************************************************************************; 

         %* (1) Check if resident is completely ADL dependent or ADL did not occur;      
         _flg_adl_tot_dependent = 0;         
         if (gg0130a1_eatg_self_admsn_cd   in ("01","09","88") and 
             gg0130c1_toilt_hygne_admsn_cd in ("01","09","88") and 
             gg0170b1_sit_admsn_cd         in ("01","09","88") and 
             gg0170c1_lyng_admsn_cd        in ("01","09","88") and 
             gg0170d1_stand_admsn_cd       in ("01","09","88") and 
             gg0170e1_chr_trnsf_admsn_cd   in ("01","09","88") and 
             gg0170f1_toilt_trnsf_admsn_cd in ("01","09","88")) 
         then _flg_adl_tot_dependent = 1;

         %* (2) Check if impairment count can be calculated: the impariment count is only valid when staff assessment of mental status is completed;
         _flg_impair_cnt = 0;
         if not missing(c1000_dcsn_mkng_cd) and not missing(b0700_self_undrstod_cd) and not missing(c0700_shrt_term_memry_cd) then _flg_impair_cnt = 1;

         %* (3) Use resident interview (BIMS) to calculate PDPM Cognitive Level;
         length cognitive_level $30.;
         if      (15>= bims_score >=13) then cognitive_level = "1. Cognitively Intact"; 
         else if (12>= bims_score >=8)  then cognitive_level = "2. Mildly Impaired";
         else if ( 7>= bims_score >=0)  then cognitive_level = "3. Moderately Impaired";
         else if (not missing(bims_score) and bims_score^=99) then put "Warning: unexpected BIMS value. BIMS value is" bims_score;

         %* (4) If resident interview is not within 0-15 (e.g. BIMS is missing or equals 99) then use staff asessment of mental status as the cognitive measure;      
         if missing(bims_score) or bims_score = 99 then do;            

            %* (5) Use staff assessment to calculate PDPM Cognitive Level;
            if missing(b0100_cmts_cd) or missing(_flg_adl_tot_dependent) then call missing(cognitive_level);
            else do;
               %* Calculate severe impairment;
               if b0100_cmts_cd = "1" and _flg_adl_tot_dependent = 1 then cognitive_level = "4. Severely Impaired";
               else if c1000_dcsn_mkng_cd = "3"                      then cognitive_level = "4. Severely Impaired";
               %* If not severely impaired, then calculate impairment counts;
               else if c1000_dcsn_mkng_cd in ("0","1","2") and _flg_impair_cnt = 1 then do;
                   %* Calculate Basic Impairment Count;
                   _impairment_count = 0;
                  if c1000_dcsn_mkng_cd       in ("1","2")     then _impairment_count = _impairment_count + 1;         
                  if b0700_self_undrstod_cd   in ("1","2","3") then _impairment_count = _impairment_count + 1;
                  if c0700_shrt_term_memry_cd = "1"            then _impairment_count = _impairment_count + 1;
                  %* Calculate Severe Impairment Count;
                  _severe_impairment_count = 0;         
                  if c1000_dcsn_mkng_cd      = "2"       then _severe_impairment_count = _severe_impairment_count + 1;
                  if b0700_self_undrstod_cd in ("2","3") then _severe_impairment_count = _severe_impairment_count + 1;
                  %* Calculate intact, mild, and moderate impairment based on Basic Impairment Count and Severe Impairment Count;
                  if _impairment_count in (2, 3) and _severe_impairment_count in (1, 2) then cognitive_level = "3. Moderately Impaired"; 
                  if _impairment_count in (2, 3) and _severe_impairment_count = 0       then cognitive_level = "2. Mildly Impaired";
                  if _impairment_count = 1                                              then cognitive_level = "2. Mildly Impaired";                     
                  if _impairment_count = 0       and _severe_impairment_count = 0       then cognitive_level = "1. Cognitively Intact";
               end;
            end;
         end;

         %*******************************************************************************************************;
         %***    Construct Comorbidity Flags;
         %***       calls create_i8000_condition_flgs macro;
         %***       creates two sets of binary variables       - SLP comorbidity flags  NTA comorbidity flags; 
         %*******************************************************************************************************; 

         %* (1) Create SLP and NTA comorbidity flags based on icd10 codes recorded on I8000 items;          
         %create_i8000_condition_flgs();  

         %*******************************************************************************************************;
         %***    Construct SLP Related Variables;
         %***       creates three intermediate grouping variables - flg_neuro  flg_slp_cmrbd  flag_impaired;
         %***       creates two final grouping variables          - neuro_cmrbd_impaired  swallow_MAD; 
         %*******************************************************************************************************;                 

         %* (1) Create a flag to indicate if an assessments clinical category belongs to Acute Neurologic;                  
         flg_neuro      = ifn(slp_clinical_category = "1. Non-Neurologic", 0, 1);  

         %* (2) Create a flag to indicate if an assessment has any slp-related comorbidities;
         flg_slp_cmrbd  = (i4300_aphasia_cd    = "1" or i4500_strk_cd             = "1" or i4900_hemiplg_cd       = "1" or 
                           i5500_brn_injury_cd = "1" or o0100e2_trchostmy_post_cd = "1" or o0100f2_vntltr_post_cd = "1" or 
                           flg_larynx = 1            or flg_apraxia                     or flg_dysphagia or flg_als or flg_oral or flg_slp_deficit);

         %* (3) Create a flag to indicate if an assessment has cognitive impairment;
         flg_impaired   = ifn(put(cognitive_level, $cognitive_format.) = "1. Cognitively Intact", 0, 1);

         %* (4) Create a combined flag to indicate if an assessment has any of the three clinical conditions constructed above;
         length neuro_cmrbd_impaired $30.;       
         if flg_neuro + flg_slp_cmrbd + flg_impaired = 0 then neuro_cmrbd_impaired = "0. None";
         if flg_neuro + flg_slp_cmrbd + flg_impaired = 1 then neuro_cmrbd_impaired = "1. Any One";  
         if flg_neuro + flg_slp_cmrbd + flg_impaired = 2 then neuro_cmrbd_impaired = "2. Any Two"; 
         if flg_neuro + flg_slp_cmrbd + flg_impaired = 3 then neuro_cmrbd_impaired = "3. All Three"; 

         %* (5) Create a combined flag to indicate if an assessment has swallowing disorder or mechanically altered diet;
         length swallow_MAD $30.;
         if k0100z_no_swlwg_cd = "0" or  k0510c2_altr_food_post_cd = "1" then swallow_MAD = "2. Either";
         if k0100z_no_swlwg_cd = "0" and k0510c2_altr_food_post_cd = "1" then swallow_MAD = "3. Both";
         if k0100z_no_swlwg_cd = "1" and k0510c2_altr_food_post_cd = "0" then swallow_MAD = "1. Neither"; 

         %*******************************************************************************************************;
         %***    Construct NTA Comorbidity Score;
         %***       redefines several MDS items to create a few NTA comorbidity flags;
         %***       creates 1 final grouping variable             - nta_cmrbd_score;
         %*******************************************************************************************************; 

         %* (1) Redefine or combine several MDS items to create certain NTA comorbidity flags; 
         %* Pressure Ulcers Stage 4;         
         m0300x1_highest_stg_ulcer_4 = ifc(m0300d1_stg_4_ulcr_num > 0, "1", "0");              
         %* Other Foot Skin Problem (M1040A or M1040C except M1040B);
         m1040ac_exc_b = ifc((m1040a_ft_infctn_cd = "1" or m1040c_othr_lsn_ft_cd = "1") and m1040b_dbtc_ft_ulcr_cd = "0", "1", "0");            
         %* Parenteral IV feeding intensity level;
         flg_PN_IV_feeding_high= (k0510a2_pen_post_cd = "1" and k0710a2_cal_prntrl_post_cd = "3");
         flg_PN_IV_feeding_low = (k0510a2_pen_post_cd = "1" and k0710a2_cal_prntrl_post_cd = "2" and k0710b2_iv_tube_daily_post_cd ="2");            

         %* (2) Calculate NTA comorbidity score;                           
         %do _n=1 %to &nnta_cmrbds.;
            %let nta_cmrbd=&&nta_cmrbds&_n..;
            %* (2.1) initiate the point to be zero for each comorbidity flag;               
            _point_&_n. = 0; 
            %* (2.2) determine point for each comorbidity if it is present on an assessment;               
            if &nta_cmrbd. then do;
               %* .find() searches the nta_cmrbd_point hash table by the key, "nta_cmrbd" (NTA comorbidity flag) from the in_assessments_dsn dataset; 
               %* If the key is found, then it merges the NTA comorbidity point assignment from the hash table;                
               if nta_cmrbd_point.find(Key:"&nta_cmrbd.")=0 then _point_&_n.= nta_point; 
            end;
         %end;            
         %* (2.3) sum up points across comorbidities;
         nta_cmrbd_score=sum(of _point:);  

         %*******************************************************************************************************;
         %****   Construct Nursing RUGs;
         %****      calls assign_nursing_rugs macro;
         %****      creates 3 intermediate grouping variables -  rug_group  flg_depression restorative_nursing; 
         %****      creates 1 final grouping variable         -  nursing_rug;
         %*******************************************************************************************************;   

         %assign_nursing_rugs();             

         ************************************************************************************************************************************;
         *                                                                                                                                   ;
         * Part 2.3: Assign Assessments to Case Mix Groups                                                                                         ;
         *                                                                                                                                   ;
         ************************************************************************************************************************************;  

         %*******************************************************************************************************;            
         %***    PT and OT case mix groups                                                                       ;  
         %*******************************************************************************************************;            
         length pt_cmg ot_cmg $200.;
         %* Only construct PT and OT case mix groups for assessments without missing PT/OT function scores and clinical category assignment; 
         %* Diagnosis codes deemed inappropriate for primary reason for SNF care are categorized as "Return to Provider" clinical category and cannot be assigned into a case mix group;         
         if cmiss(pt_ot_function_score, clinical_categories) = 0 and clinical_categories^= "Return to Provider" then do;   
            pt_cmg = catx(" | ", pt_ot_clinical_category, put(pt_ot_function_score, function_score_format.)); 
            ot_cmg = catx(" | ", pt_ot_clinical_category, put(pt_ot_function_score, function_score_format.));
         end;
         else call missing(pt_cmg, ot_cmg);

         %*******************************************************************************************************;            
         %***    SLP case mix groups                                                                             ;  
         %*******************************************************************************************************; 
         length slp_cmg $200.;            
         %* Only construct SLP case mix group for assessments without missing PDPM cognitive level, clinical categry assignment, and MDS items specific to the SLP component (mds_slp_item_list);
         %* Diagnosis codes deemed inappropriate for primary reason for SNF care are categorized as "Return to Provider" clinical category and cannot be assigned into a case mix group;                 
         if cmiss(of cognitive_level clinical_categories &mds_slp_item_list.) = 0 and clinical_categories^= "Return to Provider" then do;  
            slp_cmg = catx(" | ", neuro_cmrbd_impaired, swallow_MAD);
         end;
         else call missing(slp_cmg);

         %*******************************************************************************************************;            
         %***    NTA case mix groups                                                                             ;  
         %*******************************************************************************************************;                             
         length nta_cmg $50.;
         %* Only construct NTA case mix group for assessments without missing MDS items specific to the NTA component (mds_nta_item_list);
         %* Some NTA items are skipped and therefore have many missing values due to skip patterns in MDS (mds_nta_optional_item_list);
         %* Alternatively, if the missingness of the skippable items are redefined to default values based on the items they depend on in the input dataset;
         %* then mds_nta_optional_item_list, which contains a list of skippable items, can be added to the below cmiss function to check the missingness of all items;
         if cmiss(of &mds_nta_item_list.) = 0 then do;
            nta_cmg = put(nta_cmrbd_score, nta_score_format.);
         end;
         else call missing(nta_cmg);

         %*******************************************************************************************************;            
         %***    Nursing case mix groups                                                                         ;  
         %*******************************************************************************************************;             
         length nrsg_cmg $200.; 
         %* Only construct nursing case mix group for assessments without missing nursing function score and MDS items specific to the nursing component (mds_nursing_item_list);  
         %* Some nursing MDS items are skipped and therefore have many missing values due to skip patterns in MDS (mds_nursing_optional_item_list);
         %* Alternatively, if the missingness of the skippable items are redefined to default values based on the items they depend on in the input dataset;
         %* then mds_nursing_optional_item_list, which contains a list of skippable items, can be added to the below cmiss function to check the missingness of all items;        
         if cmiss(of &mds_nursing_item_list.) = 0 then do;            
            %do c = 1 %to &ncollapsed_rugs.;
               if findw("&&collapsed_rugs&c..",strip(nursing_rug))>0 then nrsg_cmg = "&&collapsed_rugs&c..";      
            %end;  
         end;
         else call missing(nrsg_cmg);   

         ************************************************************************************************************************************;
         *                                                                                                                                   ;
         * Part 2.4: Assign HIPPS Code to Assessments                                                                                              ;
         *                                                                                                                                   ;
         ************************************************************************************************************************************;          
         length all_cmg_short $200. hipps_code $5.;
         %* Only assign hipps code for assessments without missing cmgs;
         if cmiss(pt_cmg, ot_cmg, slp_cmg, nta_cmg, nrsg_cmg) = 0 then do;        
             _pt_ot_cmg_long    = pt_cmg;
             _slp_cmg_long      = slp_cmg;
             _nrsg_cmg_long     = nrsg_cmg;
             _nta_cmg_long      = nta_cmg;
             %* .find() searches the cmg_to_hipps hash table by the key from the in_assessments_dsn dataset; 
             %* If the key is found, then it merges the cmg_short and hipps_character variables from the hash table;                   
            if cmg_to_hipps.find(Key:_pt_ot_cmg_long) = 0 then do;
               _pt_ot_cmg_short       = cmg_short;
               _pt_ot_hipps_character = hipps_character;
            end;
            if cmg_to_hipps.find(Key:_slp_cmg_long)   = 0 then do;
               _slp_cmg_short       = cmg_short;
               _slp_hipps_character = hipps_character;
            end;
            if cmg_to_hipps.find(Key:_nrsg_cmg_long)  = 0 then do;
               _nrsg_cmg_short       = cmg_short;
               _nrsg_hipps_character = hipps_character;
            end;  
            if cmg_to_hipps.find(Key:_nta_cmg_long)   = 0 then do;
               _nta_cmg_short       = cmg_short;
               _nta_hipps_character = hipps_character;
            end;   
            %* Concatenate cmgs and assign hipps code;   
            %* assessment_indicator is an asterisk in the pdpm_sample_dataset as a placeholder;
            all_cmg_short = catx(" | ", _pt_ot_cmg_short, _slp_cmg_short, _nrsg_cmg_short, _nta_cmg_short);
            hipps_code    = catt(_pt_ot_hipps_character, _slp_hipps_character, _nrsg_hipps_character, _nta_hipps_character, assessment_indicator); 
         end;
         else call missing(all_cmg_short, hipps_code);                        
      run;

   %mend;

%****************Sample Caller ***************;
 
%assign_assessments_to_cmgs(
   
                   /* provide file path for where the main program and its accompanying macros are stored */
                   code_folder_path                 = ,
                   /* provide file path for where the assessment-level input dataset is stored */
                   input_dataset_folder_path        = ,
                   /* provide file path for where the intermediate mapping datasets are stored */
                   intermediate_mapping_folder_path = ,
                   /* provide file path for where the assessment-level output dataset will be stored */
                   output_dataset_folder_path       = ,   
                   
                   /* provide libname name followed by dataset name */
                   in_assessments_dsn   = pdpm_sample_dataset, 
                   out_assessments_dsn  = pdpm_sample_output
      
      ); 