/***************************************************************************;
%** PROGRAM: assign_nursing_rugs.sas
%** PURPOSE:  Assigns assessments to PDPM nursing RUGS.
%** DATE Created: 07/27/2018
%** NOTES: This macro is called by assign_assessments_to_cmgs.sas  
%**        This program assigns residents into nursing rugs by:
%**        1). Creates condition flags for each nursing group based on 
%**            conditions described in Chapter 6 of MDS manual.
%**        2). Classify into nursing groups based on the condition flags. 
%**        3). Assign into specific RUGs based on nursing group, function  
%**            score, depression, and # of restorative nursing services. 
%**
%**        This nursing classification logic under PDPM follows the RUG-IV    
%**        hierarchical classification logic with a couple of exceptions.
%**        PDPM uses Section GG items in place of Section G. Therefore the
%**        nursing function score and the measure of total ADL dependency
%**        is different under PDPM compared to RUG-IV.
%** 
%** RESTRICTED RIGHTS NOTICE (SEPT 2014)
%** (a) This computer software is submitted with restricted rights under Government Agreement No. CMS: DataLink 3 OY1. It may not be used, reproduced, or disclosed by the Government except as provided in paragraph (b) of this Notice or as otherwise expressly stated in the agreement. 
%** (b) This computer software may be:
%**    (1) Used or copied for use in or with the computer or computers for which it was acquired, including use at any Government installation to which such computer or computers may be transferred; 
%**    (2) Used or copied for use in a backup computer if any computer for which it was acquired is inoperative; 
%**    (3) Reproduced for safekeeping (archives) or backup purposes; 
%**    (4) Modified, adapted, or combined with other computer software, provided that the modified, combined, or adapted portions of the derivative software are made subject to the same restricted rights; 
%**    (5) Disclosed to and reproduced for use by support service Recipients in accordance with subparagraphs (b)(1) through (4) of this clause, provided the Government makes such disclosure or reproduction subject to these restricted rights; and
%**    (6) Used or copied for use in or transferred to a replacement computer. 
%** (c) Notwithstanding the foregoing, if this computer software is published copyrighted computer software, it is licensed to the Government, without disclosure prohibitions, with the minimum rights set forth in paragraph (b) of this clause. 
%** (d) Any other rights or limitations regarding the use, duplication, or disclosure of this computer software are to be expressly stated in, or incorporated in, the agreement. 
%** (e) This Notice shall be marked on any reproduction of this computer software, in whole or in part. 
%** (End of notice)  
%***************************************************************************/

%macro assign_nursing_rugs();

   *********************************************************;
   *1: Create variables and flag for classification         ;
   *********************************************************;   
   *Create Function Score Bins;
   length nursing_function_bin nursing_rug rug_group $50.;
   if      0=<  nursing_function_score <=1  then nursing_function_bin = "1. 0-1";
   else if 2<=  nursing_function_score <=5  then nursing_function_bin = "2. 2-5";
   else if 6<=  nursing_function_score <=10 then nursing_function_bin = "3. 6-10";
   else if 11<= nursing_function_score <=14 then nursing_function_bin = "4. 11-14";
   else if 15<= nursing_function_score <=16 then nursing_function_bin = "5. 15-16";  

   *Depression Flag;
   %*depression is a condition used to categorize residents into special high, special care low, and clinically complex;
   flg_depression =0; 
   if d0300_mood_scre_num >=10 and d0300_mood_scre_num ^= 99 or d0600_stf_mood_scre_num >=10 then flg_depression = 1;

   *Extensive Services conditions flag;
   _flg_ES3 = 0;
   _flg_ES2 = 0;
   _flg_ES1 = 0;
   if      o0100e2_trchostmy_post_cd =1 and o0100f2_vntltr_post_cd = 1 then _flg_ES3 = 1;
   else if o0100e2_trchostmy_post_cd =1 or  o0100f2_vntltr_post_cd = 1 then _flg_ES2 = 1;
   else if o0100m2_isltn_post_cd = 1                                   then _flg_ES1 = 1; 

   *Special Care High conditions flag;
   %*Tube feeding classification requirements (use in special care high and low);
   _flg_feeding_tube =0; 
   if k0510b1_feedg_tube_prior_cd or k0510b2_feedg_tube_post_cd then do;
      if k0710a3_cal_prntrl_7_day_cd ="3" or (k0710a3_cal_prntrl_7_day_cd ="2" and k0710b3_iv_tube_daily_7_day_cd ="2") then _flg_feeding_tube =1;   
   end;
   
   %*To classify as fever, one of the conditions must be present;
   _flg_fvr_conditions = 0;
   if i2000_pneumo_cd = "1" or j1550b_vmtg_cd = "1" or k0300_wt_loss_cd in ("1","2") or _flg_feeding_tube = "1" then _flg_fvr_conditions = 1;
       
   _flg_comatose = 0;
   _flg_ADL_tot_dependent = 0;    
   if (GG0130A1_EATG_SELF_ADMSN_CD      in ("01","09","88") 
      and GG0130C1_TOILT_HYGNE_ADMSN_CD in ("01","09","88") 
      and GG0170B1_SIT_ADMSN_CD         in ("01","09","88") 
      and GG0170C1_LYNG_ADMSN_CD        in ("01","09","88") 
      and GG0170D1_STAND_ADMSN_CD       in ("01","09","88") 
      and GG0170E1_CHR_TRNSF_ADMSN_CD   in ("01","09","88") 
      and GG0170F1_TOILT_TRNSF_ADMSN_CD in ("01","09","88")) 
   then _flg_ADL_tot_dependent = 1;
   if B0100_CMTS_CD = "1" and _flg_ADL_tot_dependent = 1 then _flg_comatose =1;

   _flg_sch_conditions         = 0;
   if _flg_comatose            = 1
      or i2100_sptcmia_cd      = "1" 
      or (i2900_dm_cd          = "1" and n0350a_insln_injct_day_num >= 7 and n0350b_insln_ordr_day_num >= 2)         
      or (i5100_quadplg_cd           and nursing_function_score <= 11) 
      or (i6200_asthma_cd      = "1" and j1100c_sob_lyg_cd = "1")
      or (j1550a_fvr_cd        = "1" and _flg_fvr_conditions =1)
      or (k0510a1_pen_prior_cd = "1" or k0510a2_pen_post_cd ="1") 
      or o0400d2_rt_day_num    >= 7 
   then _flg_sch_conditions    = 1; 

   *Special Care Low conditions flag;
   %*Selected skin treatments for ulcer care in special care low;
   if m1200a_prsr_rdc_chr_cd or m1200b_prsr_rdc_bed_cd then _pressure_relief ="1";
   else _pressure_relief ="0";
   _n_skin_treatment_scl = sum(_pressure_relief, m1200c_trng_pgm_cd, m1200d_hydrtn_cd, m1200e_ulcr_care_cd, m1200g_aplctn_drsng_cd, m1200h_aplctn_ontmnt_cd);         

   _flg_scl_conditions            = 0;      
   if (i4400_crbrl_plsy_cd        = "1" and nursing_function_score <= 11)  
      or (i5200_ms_cd             = "1" and nursing_function_score <= 11) 
      or (i5300_prknsn_cd         = "1" and nursing_function_score <= 11)     
      or (i6300_rsprtry_failr_cd  = "1" and o0100c2_oxgn_post_cd ="1")
      or _flg_feeding_tube        = 1
      or (m0300b1_stg_2_ulcr_num  >= 2  and _n_skin_treatment_scl >=2)
      or ((m0300c1_stg_3_ulcr_num >= 1  or  m0300d1_stg_4_ulcr_num >=1 or m0300f1_unstgbl_ulcr_esc_num >=1) and _n_skin_treatment_scl >=2) 
      or (m1030_artrl_ulcr_num    >= 2  and _n_skin_treatment_scl >=2)
      or (m0300b1_stg_2_ulcr_num  >= 1  and m1030_artrl_ulcr_num >=1   and _n_skin_treatment_scl >=2)
      or ((m1040a_ft_infctn_cd    = "1" or  m1040b_dbtc_ft_ulcr_cd ="1" or m1040c_othr_lsn_ft_cd ="1") and m1200i_aplctn_drsng_foot_cd ="1") 
      or o0100b2_rdtn_post_cd     = "1"
      or o0100j2_dlys_post_cd     = "1"
   then _flg_scl_conditions       = 1; 

   *Clinically Complex conditions flag;
   %*Selected skin treatments for wound care in clinically complex;   
   _n_skin_treatment_cc = sum(m1200f_srgcl_wnd_care_cd, m1200g_aplctn_drsng_cd, m1200h_aplctn_ontmnt_cd); 

   _flg_cc_conditions = 0;  
   if i2000_pneumo_cd
      or (i4900_hemiplg_cd    and nursing_function_score <= 11) 
      or ((m1040d_open_lsn_cd or m1040e_srgcl_wnd_cd) and _n_skin_treatment_cc >0)
      or m1040f_brn_cd
      or o0100a2_chmthrpy_post_cd 
      or o0100c2_oxgn_post_cd   
      or o0100h2_iv_mdctn_post_cd 
      or o0100i2_trnsfsn_post_cd
   then _flg_cc_conditions = 1;

   *Behavioral Symptoms & Cognitive Performance conditions flag;                  
   
   %*1). Calculate congnitive status based on staff assessment of mental status (Step 3);   
   %*1.1). Criteria #3 of Step 3: Calculate flags for items in the staff assessment;
   _flg_B0700 = 0;
   if B0700_SELF_UNDRSTOD_CD >0 then _flg_B0700 = 1;

   _flg_C0100 = 0;
   if C1000_DCSN_MKNG_CD >0 then _flg_C0100 = 1;

   _flg_C0700 = 0;
   if C0700_SHRT_TERM_MEMRY_CD =1 then _flg_C0700 = 1;
   
   %*1.2). Criteria #3 of Step 3: Impaired if two or more are present;
   _n_staff_impairment = sum(_flg_B0700, _flg_C0100, _flg_C0700);
   _flg_staff_impaired = 0;         
   if _n_staff_impairment >= 2 then _flg_staff_impaired = 1;
   
   %*1.3). Criteria #3 of Step 3: Severely impaired if one or more are present;
   _flg_staff_severe_impairement = 0;
   if C1000_DCSN_MKNG_CD >=2 or B0700_SELF_UNDRSTOD_CD >= 2 then _flg_staff_severe_impairement =1;
   
   %*1.4). Step 3: Final staff assessment flag - either comatose, severely impaired, or criteria 3;
   _flg_staff_final_impairment = 0;
   if _flg_comatose =1 
      or C1000_DCSN_MKNG_CD = 3 
      or (_flg_staff_impaired = 1 and _flg_staff_severe_impairement = 1) 
   then _flg_staff_final_impairment = 1;      

   %* 2). Calculate final cognitive performance based on Staff Assessment or BIMS;    
   *Cognitive Performance flag;
   _flg_cog_performance =0;         
   if bims_score<=9 or _flg_staff_final_impairment =1 then _flg_cog_performance =1;

   %* 3). Calculate behavioral issues;    
   *Behavioral Symptoms flag;
   _flg_behavior = 0;
   %do b= 1 %to &nbehavior_symp.;
      if e0100a_hllcntn_cd or e0100b_dlsn_cd or &&behavior_symp&b.. in ("2", "3") then _flg_behavior = 1;
   %end;

   *Restorative Nursing Services Count;
   %*number of resotrative nursing services is used to categorize residents into 
   %*Behavioral Symptoms & Cognitive Performance and Reduced Physical Function groups;  
   
   %*Only one of the following services count - count the one with greatest number of days;
   _nursing_toilet   = max(h0200c_crnt_toiltg_pgm_cd, h0500_bwl_toiltg_pgm_cd); 
   _nursing_mobility = max(o0500d_bed_mblty_trng_num, o0500f_wlkg_trng_num);   
   _nursing_rom      = max(o0500a_psv_rom_num, o0500b_actv_rom_num); 

   %*For Section O special treatment items, restorative nursing services count only if provided for 6 or more days;
   %do r = 1 %to &nrestorative_nursing.;
      _flg_&&restorative_nursing&r.. = 1;
      if &&restorative_nursing&r.. < 6 then _flg_&&restorative_nursing&r.. = .;
   %end;
   _n_restorative_nursing = sum(of %do r = 1 %to &nrestorative_nursing.; _flg_&&restorative_nursing&r.. %end; _nursing_toilet);
   if _n_restorative_nursing>=2 then restorative_nursing = "2 or more";
   else restorative_nursing = "0-1";  
          
   *********************************************************;
   *2: Assign residents to RUGS                             ;
   *********************************************************;   
   *Classify into nursing groups; 
   if      (_flg_ES3 or _flg_ES2 or _flg_ES1)                                               and 0=<nursing_function_score<=14  then rug_group = "Extensive Services";
   else if _flg_sch_conditions                                                              and 0=<nursing_function_score<=14  then rug_group = "Special Care High";   
   else if _flg_scl_conditions                                                              and 0=<nursing_function_score<=14  then rug_group = "Special Care Low";  
   else if _flg_cc_conditions                                                                                                  then rug_group = "Clinically Complex"; 
   else if (_flg_ES3 or _flg_ES2 or _flg_ES1 or _flg_sch_conditions or _flg_scl_conditions) and nursing_function_score>=15     then rug_group = "Clinically Complex"; 
   else if (_flg_behavior or _flg_cog_performance)                                          and 11=<nursing_function_score<=16 then rug_group = "Behavioral Cognitive Symptoms";      
   else rug_group = "Reduced Physical Function";

   *Assign Extensive Services RUGS;      
   if      _flg_ES3 and 0=<nursing_function_score<=14 then nursing_rug = "ES3"; 
   else if _flg_ES2 and 0=<nursing_function_score<=14 then nursing_rug = "ES2"; 
   else if _flg_ES1 and 0=<nursing_function_score<=14 then nursing_rug = "ES1"; 

   *Assign Special Care High RUGS;
   if rug_group = "Special Care High" and flg_depression and nursing_function_bin = "1. 0-1"   then nursing_rug = "HE2"; 
   if rug_group = "Special Care High" and flg_depression and nursing_function_bin = "2. 2-5"   then nursing_rug = "HD2"; 
   if rug_group = "Special Care High" and flg_depression and nursing_function_bin = "3. 6-10"  then nursing_rug = "HC2"; 
   if rug_group = "Special Care High" and flg_depression and nursing_function_bin = "4. 11-14" then nursing_rug = "HB2"; 

   if rug_group = "Special Care High" and flg_depression =0 and nursing_function_bin = "1. 0-1"   then nursing_rug = "HE1"; 
   if rug_group = "Special Care High" and flg_depression =0 and nursing_function_bin = "2. 2-5"   then nursing_rug = "HD1"; 
   if rug_group = "Special Care High" and flg_depression =0 and nursing_function_bin = "3. 6-10"  then nursing_rug = "HC1"; 
   if rug_group = "Special Care High" and flg_depression =0 and nursing_function_bin = "4. 11-14" then nursing_rug = "HB1";          

   *Assign Special Care Low RUGS;
   if rug_group = "Special Care Low" and flg_depression and nursing_function_bin = "1. 0-1"   then nursing_rug = "LE2"; 
   if rug_group = "Special Care Low" and flg_depression and nursing_function_bin = "2. 2-5"   then nursing_rug = "LD2"; 
   if rug_group = "Special Care Low" and flg_depression and nursing_function_bin = "3. 6-10"  then nursing_rug = "LC2"; 
   if rug_group = "Special Care Low" and flg_depression and nursing_function_bin = "4. 11-14" then nursing_rug = "LB2"; 

   if rug_group = "Special Care Low" and flg_depression =0 and nursing_function_bin = "1. 0-1"   then nursing_rug = "LE1"; 
   if rug_group = "Special Care Low" and flg_depression =0 and nursing_function_bin = "2. 2-5"   then nursing_rug = "LD1"; 
   if rug_group = "Special Care Low" and flg_depression =0 and nursing_function_bin = "3. 6-10"  then nursing_rug = "LC1"; 
   if rug_group = "Special Care Low" and flg_depression =0 and nursing_function_bin = "4. 11-14" then nursing_rug = "LB1";              

   *Assign Clinically Complex;
   if rug_group = "Clinically Complex" and flg_depression and nursing_function_bin = "1. 0-1"   then nursing_rug = "CE2"; 
   if rug_group = "Clinically Complex" and flg_depression and nursing_function_bin = "2. 2-5"   then nursing_rug = "CD2";          
   if rug_group = "Clinically Complex" and flg_depression and nursing_function_bin = "3. 6-10"  then nursing_rug = "CC2";          
   if rug_group = "Clinically Complex" and flg_depression and nursing_function_bin = "4. 11-14" then nursing_rug = "CB2"; 
   if rug_group = "Clinically Complex" and flg_depression and nursing_function_bin = "5. 15-16" then nursing_rug = "CA2";          

   if rug_group = "Clinically Complex" and flg_depression =0 and nursing_function_bin = "1. 0-1"   then nursing_rug = "CE1"; 
   if rug_group = "Clinically Complex" and flg_depression =0 and nursing_function_bin = "2. 2-5"   then nursing_rug = "CD1";          
   if rug_group = "Clinically Complex" and flg_depression =0 and nursing_function_bin = "3. 6-10"  then nursing_rug = "CC1";          
   if rug_group = "Clinically Complex" and flg_depression =0 and nursing_function_bin = "4. 11-14" then nursing_rug = "CB1"; 
   if rug_group = "Clinically Complex" and flg_depression =0 and nursing_function_bin = "5. 15-16" then nursing_rug = "CA1";  

   *Assign Behavioral Cognitive Symptoms RUGs;     
   if rug_group = "Behavioral Cognitive Symptoms" and restorative_nursing = "0-1" and nursing_function_bin = "4. 11-14" then nursing_rug = "BB1"; 
   if rug_group = "Behavioral Cognitive Symptoms" and restorative_nursing = "0-1" and nursing_function_bin = "5. 15-16" then nursing_rug = "BA1";           

   if rug_group = "Behavioral Cognitive Symptoms" and restorative_nursing = "2 or more" and nursing_function_bin = "4. 11-14" then nursing_rug = "BB2"; 
   if rug_group = "Behavioral Cognitive Symptoms" and restorative_nursing = "2 or more" and nursing_function_bin = "5. 15-16" then nursing_rug = "BA2";           

   *Assign Reduced Physical Function;           
   if rug_group = "Reduced Physical Function" and restorative_nursing = "0-1" and nursing_function_bin = "1. 0-1"   then nursing_rug = "PE1"; 
   if rug_group = "Reduced Physical Function" and restorative_nursing = "0-1" and nursing_function_bin = "2. 2-5"   then nursing_rug = "PD1";          
   if rug_group = "Reduced Physical Function" and restorative_nursing = "0-1" and nursing_function_bin = "3. 6-10"  then nursing_rug = "PC1";          
   if rug_group = "Reduced Physical Function" and restorative_nursing = "0-1" and nursing_function_bin = "4. 11-14" then nursing_rug = "PB1"; 
   if rug_group = "Reduced Physical Function" and restorative_nursing = "0-1" and nursing_function_bin = "5. 15-16" then nursing_rug = "PA1";  

   if rug_group = "Reduced Physical Function" and restorative_nursing = "2 or more" and nursing_function_bin = "1. 0-1"   then nursing_rug = "PE2"; 
   if rug_group = "Reduced Physical Function" and restorative_nursing = "2 or more" and nursing_function_bin = "2. 2-5"   then nursing_rug = "PD2";          
   if rug_group = "Reduced Physical Function" and restorative_nursing = "2 or more" and nursing_function_bin = "3. 6-10"  then nursing_rug = "PC2";          
   if rug_group = "Reduced Physical Function" and restorative_nursing = "2 or more" and nursing_function_bin = "4. 11-14" then nursing_rug = "PB2"; 
   if rug_group = "Reduced Physical Function" and restorative_nursing = "2 or more" and nursing_function_bin = "5. 15-16" then nursing_rug = "PA2";   
   
%mend;