/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.data.loader;

import com.mmm.cms.hh.pdgm.data.interf.DataLoaderIF;
import com.mmm.cms.hh.pdgm.data.interf.LineParserIF;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import com.mmm.cms.hh.pdgm.data.interf.PositionAccessorIF;
import com.mmm.cms.hh.pdgm.data.interf.RecordLayoutIF;
import java.io.FileNotFoundException;
import java.util.stream.Stream;
import static com.mmm.cms.hh.pdgm.data.meta.MetaData.ENCODING_DEFAULT;

/**
 *
 * @author US340646
 * @param <T>
 */
public class DataLoader<T> implements DataLoaderIF {

    private static final Logger LOG = Logger.getLogger(DataLoader.class.getName());

    /**
     *
     * @param clazz
     */
    public DataLoader(Class<T> clazz) {
        this(clazz, new LineParser(), false);
    }
    
    /**
     *
     * @param clazz
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, boolean loadDescriptions) {
        this(clazz, new LineParser(), loadDescriptions);
    }

    /**
     *
     * @param clazz
     * @param recordLayout
     */
    public DataLoader(Class<T> clazz, RecordLayoutIF recordLayout) {
        this(clazz, recordLayout, false);
    }
    
    /**
     *
     * @param clazz
     * @param recordLayout
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, RecordLayoutIF recordLayout, boolean loadDescriptions) {
        this(clazz, new LineParser(recordLayout.getLayout()), loadDescriptions);
    }

    /**
     *
     * @param clazz
     * @param lineParser
     */
    public DataLoader(Class<T> clazz, LineParserIF lineParser) {
        this(clazz, lineParser, false);
    }
    
    /**
     *
     * @param clazz
     * @param lineParser
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, LineParserIF lineParser, boolean loadDescriptions) {
        this.mapTo = (String line) -> {
            List<String> tokens = lineParser.parseLine(line);
            PositionAccessorIF data;
            try {
                T t = clazz.newInstance();
                data = (PositionAccessorIF) t;
                data.setLoadDescription(loadDescriptions);
                for (int i = 0; i < tokens.size(); i++) {
                    data.setByPosition(i, tokens.get(i));
                }
                return t;
            } catch (InstantiationException | IllegalAccessException ex) {
                Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
            }
            return null;
        };
    }

    /**
     *
     * @param inputFile
     * @param withHeaders
     * @return
     */
    public List<T> loadFile(InputStream inputFile, boolean withHeaders) {
        List<T> inputList = new ArrayList();
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputFile, ENCODING_DEFAULT));
            // skip the header of the tsv
            if (withHeaders) {
                inputList = br.lines().map(mapTo).collect(Collectors.toList());
            } else {
                inputList = br.lines().skip(1).map(mapTo).collect(Collectors.toList());
            }
            br.close();
        } catch (IOException ex) {
            Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
        }
        return inputList;

    }

    @Override
    public List<T> loadFile(String inputFile, boolean withHeaders) {
        File inputF = new File(inputFile);
        InputStream inputFS = null;
        try {
            inputFS = new FileInputStream(inputF);
        } catch (FileNotFoundException ex) {
            Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
        }
        return loadFile(inputFS, withHeaders);
    }

    private final Function<String, T> mapTo;

    @Override
    public List<T> loadFile(String fileName) {
        return loadFile(fileName, false);
    }

    /**
     *
     * @param inputFile
     * @return
     */
    public List<T> loadFile(InputStream inputFile) {
        return loadFile(inputFile, false);
    }

    @Override
    public T load(String s) {
        return Stream.of(s).map(mapTo).collect(Collectors.toList()).get(0);
    }

}
