/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.grouper;

import com.mmm.cms.hh.pdgm.claim.ClaimAdapter;
import com.mmm.cms.hh.pdgm.claim.ClaimIF;
import com.mmm.cms.hh.pdgm.claim.HippsCode;
import com.mmm.cms.hh.pdgm.claim.ReturnCode;
import com.mmm.cms.hh.pdgm.data.exchange.ClaimContainer;
import com.mmm.cms.hh.pdgm.data.exchange.ClaimContainerIF;
import com.mmm.cms.hh.pdgm.data.exchange.HippsContainer;
import com.mmm.cms.hh.pdgm.data.exchange.HippsContainerIF;
import com.mmm.cms.hh.pdgm.data.exchange.ReturnCodeContainer;
import com.mmm.cms.hh.pdgm.data.exchange.ValidityFlagContainer;
import com.mmm.cms.hh.pdgm.data.files.FileProperties;
import com.mmm.cms.hh.pdgm.data.files.TableNames_EN;
import com.mmm.cms.hh.pdgm.data.interf.PositionAccessorIF;
import static com.mmm.cms.hh.pdgm.data.meta.MetaData.FORMAT_DATE;
import com.mmm.cms.hh.pdgm.data.meta.enumer.ActionFlag_EN;
import com.mmm.cms.hh.pdgm.logic.processor.HippsPositionIterator;
import com.mmm.cms.hh.pdgm.logic.processor.HippsProcessor;
import com.mmm.cms.hh.pdgm.reference.objects.ClaimObject;
import com.mmm.cms.hh.pdgm.reference.objects.GrouperVersionObject;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author US340646
 */
public abstract class AbstractGrouper implements GrouperIF {

    private static final Logger LOG = Logger.getLogger(AbstractGrouper.class.getName());

    DataManagerIF dataManager;
    ClaimManagerIF claimManager;
    HippsProcessor hippsProcessor;

    /**
     *
     * @return
     */
    @Override
    public List<Integer> getClaimLayout() {
        return claimManager.getClaimLayout();
    }

    /**
     *
     * @return
     */
    @Override
    public List<Integer> getClaimLayoutInputOnly() {
        return claimManager.getClaimLayoutInputOnly();
    }

    /**
     * @return the startDate
     */
    @Override
    public Date getStartDate() {
        return new Date(startDate.getTime());
    }

    /**
     * @param startDate the startDate to set
     */
    public void setStartDate(Date startDate) {
        this.startDate = new Date(startDate.getTime());
    }

    /**
     * @return the endDate
     */
    @Override
    public Date getEndDate() {
        return new Date(endDate.getTime());
    }

    /**
     * @param endDate the endDate to set
     */
    public void setEndDate(Date endDate) {
        this.endDate = new Date(endDate.getTime());
    }

    /**
     *
     * @param id
     * @param name
     * @param description
     * @param startDate
     * @param endDate
     * @param active
     * @param defaultt
     * @throws ParseException
     */
    public AbstractGrouper(int id, String name, String description, String startDate, String endDate, boolean active, boolean defaultt) throws ParseException {

        this.id = id;
        this.name = name;
        this.description = description;

        SimpleDateFormat sdf = new SimpleDateFormat(FORMAT_DATE);

        this.startDate = sdf.parse(startDate);
        this.endDate = sdf.parse(endDate);
        this.active = active;
        this.defaultFlag = defaultt;

        // Data Manager Prototype
//        dataManager = new DataManager(buildRtBasePath()); // no descriptions
        dataManager = new DataManager(buildRtBasePath(), true); // with descriptions

        // Claim Manager Prototype
        claimManager = new ClaimManager(buildClaimLayoutPath());
        LOG.log(Level.INFO, "Grouper version {0} instantiated", this.name);
    }

    private String buildRtBasePath() {
        String retPath = FileProperties.FILE_BASE + "/"
                + this.name.replaceAll("\\.", "") + "/RT";
        return retPath;
    }

    private String buildClaimLayoutPath() {
        String retPath = FileProperties.FILE_BASE + "/"
                + this.name.replaceAll("\\.", "");
        return retPath;
    }

    /**
     *
     * @param id
     * @param grouperVersionObject
     * @throws ParseException
     */
    public AbstractGrouper(Integer id, GrouperVersionObject grouperVersionObject) throws ParseException {
        this(id,
                grouperVersionObject.getId(),
                grouperVersionObject.getDescription(),
                grouperVersionObject.getStartDate(),
                grouperVersionObject.getEndDate(),
                Integer.parseInt(grouperVersionObject.getActiveFlag()) > 0,
                Integer.parseInt(grouperVersionObject.getDefaultFlag()) > 0
        );
    }

    /**
     * @return the id
     */
    @Override
    public int getId() {
        return id;
    }

    /**
     * @param id the id to set
     */
    @Override
    public void setId(int id) {
        this.id = id;
    }

    /**
     * @return the name
     */
    @Override
    public String getName() {
        return name;
    }

    /**
     * @param name the name to set
     */
    @Override
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the description
     */
    @Override
    public String getDescription() {
        return description;
    }

    /**
     * @param description the description to set
     */
    @Override
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * @return the active
     */
    @Override
    public boolean isActive() {
        return active;
    }

    /**
     * @param active the active to set
     */
    public void setActive(boolean active) {
        this.active = active;
    }

    private int id;
    private String name;
    private String description;
    private boolean active;
    private Date startDate;
    private Date endDate;
    private boolean defaultFlag;

    /**
     * @return the defaultFlag
     */
    @Override
    public boolean isDefault() {
        return defaultFlag;
    }

    /**
     * @param defaultFlag the defaultFlag to set
     */
    public void setDefault(boolean defaultFlag) {
        this.defaultFlag = defaultFlag;
    }

    /**
     *
     * @return
     */
    @Override
    public ClaimContainerIF getEmptyClaim() {
        return new ClaimContainer();
    }

    /**
     *
     * @param claimInput
     */
    @Override
    public void group(ClaimContainerIF claimInput) {
        // convert to internal format
        ClaimIF claim = (new ClaimAdapter()).convert(claimInput);
        group(claim);

        // populate results back
        claimInput.setVersionUsed(claim.getGrouperVersion());
//        claimInput.setHippsCode(claim.getHippsCode().getHippsCode()); /* do not populate incomplete HIPPS */
        claimInput.setHippsCode(claim.getHippsCode().getFinalHippsCode());
        // populate HIPPS object (need to move that stuff to ClaimAdapter class
        HippsContainerIF hipps = new HippsContainer();
        if (claim.getHippsCode().isHippsComplete()) {
            for (int i = 1; i < hipps.getPositionsCount() + 1; i++) {
                String value = claim.getHippsCode().getValue(i).getValue();
                String posDesc = claim.getHippsCode().getValue(i).getPositionDescription();
                String valDesc = claim.getHippsCode().getValue(i).getValueDescription();
                hipps.setPositionValue(i, value);
                hipps.setPositionDescription(i, posDesc + ": " + valDesc);
            }
        }
        claimInput.setHippsContainer(hipps);
        // set validity
        claimInput.setValidityFlag(
                new ValidityFlagContainer(claim.getSingleValidityFlag().getValue(),
                        claim.getSingleValidityFlag().getDescription())
        );
        // set return code
        claimInput.setReturnCode(
                new ReturnCodeContainer(String.format("%02d", claim.getSingleReturnCode().getId()),
                        claim.getSingleReturnCode().getDescription())
        );
        // set edits
        claimInput.setEdits(claim.getClaimEdits());
    }

    /**
     *
     * @param claim
     */
    protected void group(ClaimIF claim) {
        final int GRC_INTERNAL_ERROR = 50;
        // process claim
        if (claim == null) { // no null claims allowed
            return;
        }
        try {
            hippsProcessor = new HippsProcessor(claim, dataManager);
            hippsProcessor.process(new HippsPositionIterator());
            // populate version
            claim.setGrouperVersion(getVersion());
        } catch (Exception e) {
            e.printStackTrace();
            claim.getReturnCodes().clear();
            claim.getReturnCodes().add(new ReturnCode(GRC_INTERNAL_ERROR, "Grouper Internal Error"));
            claim.setHippsCode(new HippsCode());
            claim.setGrouperVersion(getVersion());
            System.out.println("*** SEVERE: Grouper Internal Error with processing claim ***");
            // Logging for internal error
            claim.getClaimEdits().getEdits().forEach((edit) -> {
                System.out.println(edit);
            });
        }
    }

    // Batch & MF grouping
    /**
     *
     * @param claimInput
     * @return
     */
    @Override
    public String group(String claimInput) {
        // read input claim
        ClaimObject claimIn = claimManager.loadClaim(claimInput);
        // convert to internal format
        ClaimIF claim = (new ClaimAdapter()).convert(claimIn);
        // process claim
        group(claim);
        // format results
        String retStr = formatResult(claim);

        // populate back results
        // Version used
        claimIn.setVersionUsed(claim.getGrouperVersion());
        // HIPPS
        claimIn.setHippsCode(claim.getHippsCode().getHippsCode());
        // Validity flag
        claimIn.setByPosition(55, claim.getSingleValidityFlag().getValue());
        // Return Code
        claimIn.setByPosition(56, String.format("%02d", claim.getSingleReturnCode().getId()));

        // Print logs
        if (claimIn.getOptions().getActionFlag().equalsIgnoreCase(ActionFlag_EN.Y.name())) {
            System.out.println("*** Grouping Detail Report ***");
            System.out.println(claimIn);
            System.out.println("*** Output string = " + retStr);
            claim.getClaimEdits().getEdits().forEach((edit) -> {
                System.out.println(edit);
            });
        }

        return retStr;
    }

    /**
     *
     * @param claim
     * @return
     */
    protected String formatResult(ClaimIF claim) {
        String retStr = getVersion()
                + claim.getHippsCode().getFinalHippsCode()
                + claim.getSingleValidityFlag().getValue()
                + String.format("%02d", claim.getSingleReturnCode().getId());
        return retStr;
    }

    /**
     *
     * @return
     */
    @Override
    public String getVersion() {
        return getName();
    }

    /**
     *
     * @param tableType
     * @return
     */
    @Override
    public DataContainerIF<PositionAccessorIF> getAvailableTableContent(TableNames_EN tableType) {
        return dataManager.getDataMap().get(tableType);
    }
}
