/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.logic.processor;

import com.mmm.cms.hh.pdgm.claim.ClaimIF;
import com.mmm.cms.hh.pdgm.data.files.TableNames_EN;
import com.mmm.cms.hh.pdgm.data.meta.enumer.ComorbidityLevel_EN;
import com.mmm.cms.hh.pdgm.data.meta.enumer.EditId_EN;
import com.mmm.cms.hh.pdgm.diagnosis.ComorbidityGroup;
import com.mmm.cms.hh.pdgm.diagnosis.ComorbidityInteraction;
import com.mmm.cms.hh.pdgm.diagnosis.ComorbidityInteractionIF;
import com.mmm.cms.hh.pdgm.diagnosis.DiagnosisIF;
import com.mmm.cms.hh.pdgm.grouper.DataManagerIF;
import com.mmm.cms.hh.pdgm.logic.specification.IsInComorbidityInteraction;
import com.mmm.cms.hh.pdgm.reference.objects.ComorbidityInteractionObject;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author us340646
 */
public class HippsPosition4 extends HippsPositionAbstract implements HippsPositionIF {

    /*
    * 4. HIPPS 4th position
    * 4.1 Validate SDX(s)
    * 4.2 Calculate 4th value
    * Note: DX validation could be performed during
     */

    /**
     *
     * @param claim
     * @param dataManager
     */

    public HippsPosition4(ClaimIF claim, DataManagerIF dataManager) {
        super(claim, dataManager);
    }

    /**
     *
     * @param visitor
     */
    @Override
    public void process(HippsPositionSelectorIF visitor) {
        visitor.select(this);
    }

    /**
     *
     * @return
     */
    @Override
    public boolean validate() {
        // Nothing to validate so far
        // Everything should be validated before
        return true;
    }

    /**
     *
     * @return
     */
    @Override
    public boolean calculate() {

        final int HIPPS_POSITION = 4;

        ComorbidityLevel_EN hippsValue = ComorbidityLevel_EN.NO_COMORBIDITY;

        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();
        List<DiagnosisIF> sdxList = getClaim().getSecondaryDiagnoses();
        // Get valid SDXs with cormobidity and diff from PDX subchapter
        List<DiagnosisIF> sdxComorbidityList = new ArrayList<>();
        for (DiagnosisIF dx : sdxList) {
            // SDX must be valid
            if (dx.getEditCollection().hasEditId(EditId_EN.VALUE_VALID)) {
                // SDX must have Comorbidity
                if (dx.getComorbidityGroup().isComorbidityPresent()) {
                    // SDX must have different from PDX subchapter
                    String logText = "SDX = " + dx.getValue() + " (" + dx.getDescription()
                                    + ") with Comorbidity Group = " + dx.getComorbidityGroup().getName()
                                    + " is eligible for comorbidity determination";
                    if (!pdx.getEditCollection().hasCriticalEdits()) {
                        if(dx.getDiagnosisSubchapter().getId() != pdx.getDiagnosisSubchapter().getId()) {
                            sdxComorbidityList.add(dx);
                            addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, logText);
                        } else {
                            String subChapter = dx.getDiagnosisSubchapter().getDxFrom() + "-"
                                    + dx.getDiagnosisSubchapter().getDxTo() + " "
                                    + dx.getDiagnosisSubchapter().getDescription();
                            addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "SDX = " + dx.getValue() + " (" + dx.getDescription()
                                    + ") has comorbidity but from the same DX subchapter as PDX ("
                                    + subChapter + "). Not eligible for comorbidity determination");
                        }
                    } else {
                        sdxComorbidityList.add(dx);
                        addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, logText
                            + " but PDX is invalid or ungroupable and HIPPS value won't be accurate");
                    }
                } else {
                    addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "SDX = " + dx.getValue() + " (" + dx.getDescription() + ")"
                            + ") has no comorbidity");
                }
            } else {
                addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "SDX = " + dx.getValue()
                        + " is invalid DX");
            }
        }

        // check comorbidity
        if (!sdxComorbidityList.isEmpty()) {
            // check for high comorbidity
            List<ComorbidityInteractionIF> allInteractionList
                    = getInitalComorbidityInteractionList();
            List<ComorbidityInteractionIF> resultInteractionList
                    = checkComorbidityInteraction(allInteractionList, sdxComorbidityList);
            // if list is not empty then high comorbidity
            if (!resultInteractionList.isEmpty()) {
                hippsValue = ComorbidityLevel_EN.HIGH_COMORBIDITY;
                resultInteractionList.forEach((cInter) -> {
                    addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "Comorbidity interaction = "
                            + cInter.getId() + ", "
                            + cInter.getComorbidityGroup1().getName()
                            + " (" + "SDX=" + findDxByComorbidity(sdxList,
                                    cInter.getComorbidityGroup1().getName()).getValue() + ")"
                            + " / "
                            + cInter.getComorbidityGroup2().getName()
                            + " (" + "SDX=" + findDxByComorbidity(sdxList,
                                    cInter.getComorbidityGroup2().getName()).getValue() + ")"
                            + " is used for high comorbidity adjustment"
                    );
                });
                addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "Comorbidity interaction(s) found for high comorbidity adjustment");
            } else {
                addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "No high comorbidity (interactions) found in SDX list");
                // check for low comorbidity
                if (isLowComorbidity(sdxComorbidityList)) {
                    hippsValue = ComorbidityLevel_EN.LOW_COMORBIDITY;
                }
            }
        }

        addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "HIPPS Position 4 = " + hippsValue.getId()
                + " (" + hippsValue.getDescription() + ")");

        // Set HIPPS 4
        return setHippsCode(HIPPS_POSITION, "" + hippsValue.getId());

    }

    private boolean isLowComorbidity(List<DiagnosisIF> dxList) {
        for (DiagnosisIF dx : dxList) {
            if (dx.getComorbidityGroup().isLowComorbidity()) {
                addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "First SDX found for low comorbidity adjustment = " + dx.getValue());
                return true;
            }
        }
        addHippsLogicMessage(EditId_EN.VALUE_HIPPS4, "No low comorbidity found in SDX list");
        return false;
    }

    private List<ComorbidityInteractionIF> checkComorbidityInteraction(List<ComorbidityInteractionIF> ciList, List<DiagnosisIF> dxList) {
        List<ComorbidityInteractionIF> ciLeftList = checkComorbidityInteraction(ciList, dxList, true);
        List<ComorbidityInteractionIF> ciRightList = checkComorbidityInteraction(ciLeftList, dxList, false);
        return ciRightList;
    }

    private List<ComorbidityInteractionIF> checkComorbidityInteraction(List<ComorbidityInteractionIF> ciList, List<DiagnosisIF> dxList, boolean isLeft) {
        List<ComorbidityInteractionIF> retList = new ArrayList<>();
        IsInComorbidityInteraction check = new IsInComorbidityInteraction(dxList);
        ciList.forEach((ci) -> {
            ComorbidityGroup cg = (isLeft ? ci.getComorbidityGroup1() : ci.getComorbidityGroup2());
            if (check.isSatisfiedBy(cg)) {
                retList.add(ci);
            }
        });
        return retList;
    }

    private List<ComorbidityInteractionIF> getInitalComorbidityInteractionList() {
        List<ComorbidityInteractionIF> retList = new ArrayList();
        List<ComorbidityInteractionObject> ciDtoList
                = getDataManager().getDataMap().get(TableNames_EN.Comorbidity_Interactions).getData();
        for (ComorbidityInteractionObject ciObj : ciDtoList) {
            ComorbidityGroup cg1 = new ComorbidityGroup(0, ciObj.getGroup1(), null, false);
            ComorbidityGroup cg2 = new ComorbidityGroup(0, ciObj.getGroup2(), null, false);
            ComorbidityInteractionIF ci = new ComorbidityInteraction(Integer.parseInt(ciObj.getId()), cg1, cg2);
            retList.add(ci);
        }
        return retList;
    }

}
