package gov.cms.fiss.pricers.ipps.core;

import gov.cms.fiss.pricers.common.api.InternalPricerException;
import gov.cms.fiss.pricers.common.api.YearNotImplementedException;
import gov.cms.fiss.pricers.common.application.ClaimProcessor;
import gov.cms.fiss.pricers.common.application.PricerDispatch;
import gov.cms.fiss.pricers.ipps.IppsPricerConfiguration;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.core.tables.DataTables;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

public class IppsPricerDispatch
    extends PricerDispatch<IppsInput, IppsOutput, IppsPricerConfiguration> {
  public IppsPricerDispatch(IppsPricerConfiguration pricerConfiguration) {
    super(pricerConfiguration, o -> Integer.toString(o.getRecord().getRtc()));
  }

  @Override
  protected void initializeReferences(IppsPricerConfiguration pricerConfiguration) {
    for (final int supportedYear : pricerConfiguration.getSupportedYears()) {
      switch (supportedYear) {
        case 2019:
          yearReference.register(
              supportedYear, Ipps2019RulePricer.class, DataTables.forYear(supportedYear));

          break;
        case 2020:
          yearReference.register(
              supportedYear, Ipps2020RulePricer.class, DataTables.forYear(supportedYear));

          break;
        case 2021:
          yearReference.register(
              supportedYear, Ipps2021RulePricer.class, DataTables.forYear(supportedYear));

          break;
        case 2022:
          yearReference.register(
              supportedYear, Ipps2022RulePricer.class, DataTables.forYear(supportedYear));

          break;
        default:
          break;
      }
    }
  }

  @Override
  protected ClaimProcessor<IppsInput, IppsOutput> getProcessor(IppsInput input)
      throws YearNotImplementedException, InternalPricerException {
    return yearReference.fromFiscalYear(
        DateTimeFormatter.BASIC_ISO_DATE.parse(
            input.getBillingRecord().getDischargeDate(), LocalDate::from),
        "dischargeDate");
  }

  @Override
  protected boolean isErrorOutput(IppsOutput output) {
    return ResultCode.isErrorCode(output.getRecord().getRtc());
  }
}
