package gov.cms.fiss.pricers.ipps.core.rules.calculate_payment.tech_addons;

import gov.cms.fiss.pricers.common.application.rules.CalculationRule;
import gov.cms.fiss.pricers.common.util.BigDecimalUtils;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.api.ProspectivePaymentRecord;
import gov.cms.fiss.pricers.ipps.core.IppsPricerContext;
import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * Determines the initial operating payment amounts and adjustments.
 *
 * <p>Converted from {@code 4000-CALC-TECH-ADDON} in the COBOL code (continued).
 *
 * @since 2019
 */
public class InitializeOperatingCosts
    implements CalculationRule<IppsInput, IppsOutput, IppsPricerContext> {

  @Override
  public void priceClaim(IppsPricerContext calculationContext) {
    final ProspectivePaymentRecord prospectivePaymentRecord =
        calculationContext.getProspectivePaymentRecord();

    // ***********************************************************
    // ***  CALCULATE TOTALS FOR OPERATING  ADD ON FOR TECH
    //     COMPUTE PPS-OPER-HSP-PART ROUNDED =
    //         H-OPER-HSP-PCT * H-OPER-HSP-PART.
    prospectivePaymentRecord.setOperHspPart(
        calculationContext
            .getOperatingHospitalSpecificPortionPct()
            .multiply(calculationContext.getOperatingHospitalSpecificPortionPart())
            .setScale(2, RoundingMode.HALF_UP));

    //     COMPUTE PPS-OPER-FSP-PART ROUNDED =
    //         H-OPER-FSP-PCT * H-OPER-FSP-PART.
    prospectivePaymentRecord.setOperFspPart(
        calculationContext
            .getOperatingFederalSpecificPortionPct()
            .multiply(calculationContext.getOperatingFederalSpecificPortionPart())
            .setScale(2, RoundingMode.HALF_UP));

    //     MOVE ZERO TO PPS-OPER-DSH-ADJ.
    prospectivePaymentRecord.setOperDshAdj(BigDecimal.ZERO);

    //     IF  H-OPER-DSH NUMERIC
    //             COMPUTE PPS-OPER-DSH-ADJ ROUNDED =
    //             (PPS-OPER-FSP-PART
    //              * H-OPER-DSH) * .25.
    prospectivePaymentRecord.setOperDshAdj(
        prospectivePaymentRecord
            .getOperFspPart()
            .multiply(calculationContext.getOperatingDisproportionateShare())
            .multiply(new BigDecimal("0.25"))
            .setScale(2, RoundingMode.HALF_UP));

    //     COMPUTE PPS-OPER-IME-ADJ ROUNDED =
    //             PPS-OPER-FSP-PART *
    //             H-OPER-IME-TEACH.
    prospectivePaymentRecord.setOperImeAdj(
        prospectivePaymentRecord
            .getOperFspPart()
            .multiply(calculationContext.getOperatingIndirectMedicalEducation())
            .setScale(2, RoundingMode.HALF_UP));

    //     COMPUTE H-BASE-DRG-PAYMENT ROUNDED =
    //             PPS-OPER-FSP-PART +
    //             PPS-OPER-DSH-ADJ + PPS-OPER-IME-ADJ +
    //             WK-UNCOMP-CARE-AMOUNT.
    calculationContext.setBaseDrgPayment(
        BigDecimalUtils.decimalSum(
                prospectivePaymentRecord.getOperFspPart(),
                prospectivePaymentRecord.getOperDshAdj(),
                prospectivePaymentRecord.getOperImeAdj(),
                calculationContext.getUncompensatedCareAmount())
            .setScale(2, RoundingMode.HALF_UP));
  }
}
