package gov.cms.fiss.pricers.ipps.core.rules.calculate_payment.totals;

import gov.cms.fiss.pricers.common.application.rules.CalculationRule;
import gov.cms.fiss.pricers.common.util.BigDecimalUtils;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.api.ProspectivePaymentRecord;
import gov.cms.fiss.pricers.ipps.api.ProviderRecord;
import gov.cms.fiss.pricers.ipps.core.IppsPricerContext;
import java.math.BigDecimal;
import java.math.RoundingMode;
import org.apache.commons.lang3.StringUtils;

/**
 * Determine the low volume total amounts.
 *
 * <p>Converted from {@code 3800-CALC-TOT-AMT} in the COBOL code.
 *
 * @since 2019
 */
public class CalculateLowVolumeTotals
    implements CalculationRule<IppsInput, IppsOutput, IppsPricerContext> {

  @Override
  public void priceClaim(IppsPricerContext calculationContext) {
    final ProspectivePaymentRecord prospectivePaymentRecord =
        calculationContext.getProspectivePaymentRecord();

    // ***********************************************************
    // *LOW VOLUME CALCULATIONS
    // ***********************************************************
    // *---------------------------------------------------------*
    // * (YEARCHANGE 2016.0)
    // * LOW VOLUME PAYMENT ADD-ON PERCENT
    // *---------------------------------------------------------*
    //     MOVE ZERO TO PPS-OPER-DSH-ADJ.
    prospectivePaymentRecord.setOperDshAdj(BigDecimal.ZERO);

    // ************************************************
    // * FOR FY 2014 WE APPLY AN ADJUSTMENT OF 0.25 TO CALCULATE
    // * EMPERICAL DSH
    // ************************************************
    //     IF  H-OPER-DSH NUMERIC
    //         COMPUTE PPS-OPER-DSH-ADJ ROUNDED =
    //                     (PPS-OPER-FSP-PART  * H-OPER-DSH) * .25.
    prospectivePaymentRecord.setOperDshAdj(
        prospectivePaymentRecord
            .getOperFspPart()
            .multiply(calculationContext.getOperatingDisproportionateShare())
            .multiply(new BigDecimal("0.25"))
            .setScale(2, RoundingMode.HALF_UP));

    //     COMPUTE PPS-OPER-IME-ADJ ROUNDED =
    //                         PPS-OPER-FSP-PART * H-OPER-IME-TEACH.
    prospectivePaymentRecord.setOperImeAdj(
        prospectivePaymentRecord
            .getOperFspPart()
            .multiply(calculationContext.getOperatingIndirectMedicalEducation())
            .setScale(2, RoundingMode.HALF_UP)); // used

    //     COMPUTE PPS-OPER-FSP-PART ROUNDED =
    //                           H-OPER-FSP-PART * H-OPER-FSP-PCT.
    prospectivePaymentRecord.setOperFspPart(
        calculationContext
            .getOperatingFederalSpecificPortionPart()
            .multiply(calculationContext.getOperatingFederalSpecificPortionPct())
            .setScale(2, RoundingMode.HALF_UP));
    //     COMPUTE PPS-OPER-HSP-PART ROUNDED =
    //                           H-OPER-HSP-PART * H-OPER-HSP-PCT.
    prospectivePaymentRecord.setOperHspPart(
        calculationContext
            .getOperatingHospitalSpecificPortionPart()
            .multiply(calculationContext.getOperatingHospitalSpecificPortionPct())
            .setScale(2, RoundingMode.HALF_UP));
    //     COMPUTE PPS-OPER-OUTLIER-PART ROUNDED =
    //                         H-OPER-OUTLIER-PART * H-OPER-FSP-PCT.
    prospectivePaymentRecord.setOperOutlierPart(
        calculationContext
            .getOperatingOutlierPart()
            .multiply(calculationContext.getOperatingFederalSpecificPortionPct())
            .setScale(2, RoundingMode.HALF_UP));

    BigDecimal lowVolAddon = BigDecimal.ZERO;
    final ProviderRecord providerRecord = calculationContext.getProviderRecord();

    //     IF P-NEW-TEMP-RELIEF-IND = 'Y'
    //        AND P-LV-ADJ-FACTOR > 0.00
    //        AND P-LV-ADJ-FACTOR <= 0.25
    //     COMPUTE WK-LOW-VOL-ADDON ROUNDED =
    //       (PPS-OPER-HSP-PART +
    //        PPS-OPER-FSP-PART +
    //        PPS-OPER-IME-ADJ +
    //        PPS-OPER-DSH-ADJ +
    //        PPS-OPER-OUTLIER-PART +
    //        H-CAPI-FSP +
    //        H-CAPI-IME-ADJ +
    //        H-CAPI-DSH-ADJ +
    //        H-CAPI-OUTLIER +
    //        WK-UNCOMP-CARE-AMOUNT +
    //        PPS-NEW-TECH-PAY-ADD-ON) * P-LV-ADJ-FACTOR
    //     ELSE
    //     COMPUTE WK-LOW-VOL-ADDON ROUNDED = 0.
    if (StringUtils.equals(providerRecord.getTemporaryReliefIndicator(), "Y")
        && BigDecimalUtils.isGreaterThanZero(providerRecord.getLvAdjustmentFactor())
        && BigDecimalUtils.isLessThanOrEqualTo(
            providerRecord.getLvAdjustmentFactor(), new BigDecimal("0.25"))) {
      lowVolAddon =
          BigDecimalUtils.decimalSum(
                  prospectivePaymentRecord.getOperHspPart(),
                  prospectivePaymentRecord.getOperFspPart(),
                  prospectivePaymentRecord.getOperImeAdj(),
                  prospectivePaymentRecord.getOperDshAdj(),
                  prospectivePaymentRecord.getOperOutlierPart(),
                  calculationContext.getCapitalFederalSpecificPortion(),
                  calculationContext.getCapitalIndirectMedicalEducationAdj(),
                  calculationContext.getCapitalDisproportionateShareHospitalAdjustment(),
                  calculationContext.getCapitalOutlierCost(),
                  calculationContext.getUncompensatedCareAmount(),
                  calculationContext.getNewTechAddOnPayment())
              .multiply(providerRecord.getLvAdjustmentFactor())
              .setScale(2, RoundingMode.HALF_UP);
    }

    //     COMPUTE H-LOW-VOL-PAYMENT ROUNDED = WK-LOW-VOL-ADDON.
    calculationContext.setLowVolumePayment(lowVolAddon);
  }
}
