package gov.cms.fiss.pricers.ipps.core.rules.rules2021.wage_index;

import gov.cms.fiss.pricers.common.application.rules.CalculationRule;
import gov.cms.fiss.pricers.common.util.BigDecimalUtils;
import gov.cms.fiss.pricers.ipps.api.CbsaData;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.core.CbsaReference;
import gov.cms.fiss.pricers.ipps.core.IppsPricerContext;
import gov.cms.fiss.pricers.ipps.core.ResultCode;
import java.math.RoundingMode;
import org.apache.commons.lang3.StringUtils;

/**
 * Determines whether the wage index value will undergo a previous-year adjustment.
 *
 * <p>Converted from {@code 0550-GET-CBSA} in the COBOL code (continued).
 *
 * @since 2020
 */
public class AdjustIndexForPreviousYear2021
    implements CalculationRule<IppsInput, IppsOutput, IppsPricerContext> {

  @Override
  public void priceClaim(IppsPricerContext calculationContext) {
    final CbsaReference cbsaReference = calculationContext.getCbsaReference();
    final CbsaData cbsaData = calculationContext.getCbsaData();

    // **----------------------------------------------------------------
    // ** APPLY RURAL FLOOR POLICY TRANSITION ADJUSTMENT
    // **----------------------------------------------------------------
    //
    //     IF B-N-DISCHARGE-DATE > 20200930
    //        IF P-NEW-IND-PRIOR-YEAR AND P-NEW-SUPP-WI > 0
    //           MOVE P-NEW-SUPP-WI TO HLD-PREV-WI
    //        ELSE
    //           MOVE 52 TO PPS-RTC
    //           GOBACK.
    if (!StringUtils.equals(cbsaData.getSupplementalWageIndexIndicator(), "1")
        || BigDecimalUtils.isLessThanOrEqualToZero(cbsaData.getSupplementalWageIndex())) {
      calculationContext.applyResultCode(ResultCode.RC_52_INVALID_WAGE_INDEX);
      return;
    }

    //     IF B-N-DISCHARGE-DATE > 20190930
    //        IF (((W-NEW-CBSA-WI - HLD-PREV-WI) / HLD-PREV-WI)
    //              < WI_PCT_REDUC_FY2020)
    //           COMPUTE W-NEW-CBSA-WI ROUNDED =
    //              HLD-PREV-WI * WI_PCT_ADJ_FY2020.
    if (BigDecimalUtils.isLessThan(
        cbsaReference
            .getWageIndex()
            .subtract(cbsaData.getSupplementalWageIndex())
            .divide(cbsaData.getSupplementalWageIndex(), RoundingMode.HALF_UP),
        calculationContext.getWageIndexPctReduction())) {
      cbsaReference.setWageIndex(
          cbsaData
              .getSupplementalWageIndex()
              .multiply(calculationContext.getWageIndexPctAdj())
              .setScale(4, RoundingMode.HALF_UP));
    }
  }
}
