package gov.cms.fiss.pricers.ipps.core.rules.rules2022.assemble_variables;

import gov.cms.fiss.pricers.common.application.rules.CalculationRule;
import gov.cms.fiss.pricers.common.util.BigDecimalUtils;
import gov.cms.fiss.pricers.ipps.api.BillingRecord;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.core.IppsPricerContext;
import gov.cms.fiss.pricers.ipps.core.rules.rules2021.assemble_variables.DetermineNewCovid19TreatmentsAddOnPayment2021;
import gov.cms.fiss.pricers.ipps.core.tables.ClaimCodeType;
import gov.cms.fiss.pricers.ipps.core.tables.DataTables;
import java.math.BigDecimal;

/**
 * Determine the claim's eligibility for an adjustment to the Operating Base DRG Payment and the New
 * Tech Add-On Payment based on a COVID-19 diagnosis/procedure/condition code. A unique calculated
 * factor is effective as of 11/02/2020, 11/19/2020, or 01/01/2021.
 *
 * <p>Converted from {@code 3650-NEW-COVID19-ADD-ON-PAY} in the COBOL code.
 *
 * @since 2021
 */
public class DetermineNewCovid19TreatmentsAddOnPayment2022
    extends DetermineNewCovid19TreatmentsAddOnPayment2021
    implements CalculationRule<IppsInput, IppsOutput, IppsPricerContext> {

  /**
   * Converted from {@code 3650-NEW-COVID19-ADD-ON-PAY} in the COBOL code.
   *
   * @param calculationContext the current pricing context
   */
  @Override
  protected void newCovid19TreatmentsAddOnPaymentValue(IppsPricerContext calculationContext) {
    final DataTables dataTables = calculationContext.getDataTables();
    final BillingRecord billingRecord = calculationContext.getBillingRecord();

    //      MOVE 0 TO H-LESSER-STOP-1
    //                H-LESSER-STOP-2.
    //
    //      COMPUTE H-LESSER-STOP-1 ROUNDED =
    //          H-OPER-DOLLAR-THRESHOLD * 0.65.
    //
    //      COMPUTE H-LESSER-STOP-2 ROUNDED =
    //          (H-OPER-BILL-COSTS - (H-OPER-COST-OUTLIER -
    //              H-OPER-DOLLAR-THRESHOLD)) * 0.65.

    final BigDecimal lesserNCTAP1 =
        calculationContext.getOperatingDollarThreshold().multiply(new BigDecimal("0.65"));

    final BigDecimal lesserNCTAP2;

    if (dataTables.codesMatch("VEKLURY", ClaimCodeType.PROC, billingRecord.getProcedureCodes())) {
      lesserNCTAP2 =
          calculationContext
              .getOperatingBillCosts()
              .subtract(
                  calculationContext
                      .getOperatingCostOutlier()
                      .subtract(calculationContext.getOperatingDollarThreshold())
                      .subtract(BigDecimal.valueOf(2028.00)))
              .multiply(new BigDecimal("0.65"));
    } else {
      lesserNCTAP2 =
          calculationContext
              .getOperatingBillCosts()
              .subtract(
                  calculationContext
                      .getOperatingCostOutlier()
                      .subtract(calculationContext.getOperatingDollarThreshold()))
              .multiply(new BigDecimal("0.65"));
    }

    //      IF H-OPER-BILL-COSTS >
    //          (H-OPER-COST-OUTLIER - H-OPER-DOLLAR-THRESHOLD)
    //          IF H-LESSER-STOP-1 < H-LESSER-STOP-2
    //              MOVE H-LESSER-STOP-1 TO NCTAP-ADD-ON
    //          ELSE
    //              MOVE H-LESSER-STOP-2 TO NCTAP-ADD-ON
    //      ELSE
    //          MOVE ZEROES TO NCTAP-ADD-ON.
    //
    //      MOVE 0 TO H-LESSER-STOP-1
    //                H-LESSER-STOP-2.

    if (BigDecimalUtils.isGreaterThan(
        calculationContext.getOperatingBillCosts(),
        calculationContext
            .getOperatingCostOutlier()
            .subtract(calculationContext.getOperatingDollarThreshold()))) {
      if (BigDecimalUtils.isLessThan(lesserNCTAP1, lesserNCTAP2)) {
        calculationContext.setNewCovid19TreatmentsAddOnPayment(lesserNCTAP1);
      } else {
        calculationContext.setNewCovid19TreatmentsAddOnPayment(lesserNCTAP2);
      }
    }

    if (dataTables.codesMatch("VEKLURY", ClaimCodeType.PROC, billingRecord.getProcedureCodes())) {
      calculationContext.setNewCovid19TreatmentsAddOnPayment(
          calculationContext
              .getNewCovid19TreatmentsAddOnPayment()
              .subtract(BigDecimal.valueOf(2028.00)));
    }

    if (BigDecimalUtils.isGreaterThan(
        calculationContext.getNewCovid19TreatmentsAddOnPayment(), BigDecimal.ZERO)) {
      calculationContext.setNewCovid19TreatmentsAddOnPayment(
          calculationContext.getNewCovid19TreatmentsAddOnPayment());
    } else {
      calculationContext.setNewCovid19TreatmentsAddOnPayment(BigDecimal.ZERO);
    }
  }
}
