package gov.cms.fiss.pricers.ipps.resources;

import com.codahale.metrics.annotation.Timed;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.dataformat.csv.CsvSchema;
import com.fasterxml.jackson.dataformat.csv.CsvSchema.ColumnType;
import gov.cms.fiss.pricers.common.application.resources.BaseCsvResource;
import gov.cms.fiss.pricers.ipps.IppsPricerConfiguration;
import gov.cms.fiss.pricers.ipps.api.CbsaRetrievalResponse;
import gov.cms.fiss.pricers.ipps.api.CbsaTableEntry;
import gov.cms.fiss.pricers.ipps.api.CbsaTableEntry.Fields;
import gov.cms.fiss.pricers.ipps.core.tables.DataTables;
import io.dropwizard.jersey.jsr310.LocalDateParam;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

@Path("/cbsa-wage-indexes")
public class IppsCbsaWageIndexResource extends BaseCsvResource<CbsaTableEntry> {
  public IppsCbsaWageIndexResource(IppsPricerConfiguration pricerConfiguration) {
    super(
        pricerConfiguration,
        () ->
            (CsvMapper)
                DEFAULT_CSV_MAPPER
                    .copy()
                    .addMixIn(CbsaTableEntry.class, CbsaTableEntryCsvFormatter.class),
        csvMapper ->
            CsvSchema.builder()
                .addColumn(Fields.CBSA)
                .addColumn(Fields.SIZE)
                .addColumn(Fields.EFFECTIVE_DATE, ColumnType.STRING)
                .addColumn(Fields.GEOGRAPHIC_WAGE_INDEX, ColumnType.NUMBER)
                .addColumn(Fields.RECLASSIFIED_WAGE_INDEX, ColumnType.NUMBER)
                .addColumn(Fields.RURAL_FLOOR_WAGE_INDEX, ColumnType.NUMBER)
                .addColumn(Fields.IMPUTED_FLOOR_WAGE_INDEX, ColumnType.NUMBER)
                .addColumn(Fields.NAME)
                .build()
                .withHeader());
  }

  @Override
  protected void extractAndPopulate(CsvMapper csvMapper, CsvSchema csvSchema, Integer pricerYear) {
    final Optional<Map<String, NavigableMap<LocalDate, CbsaTableEntry>>> matchingData =
        Optional.ofNullable(DataTables.forYear(pricerYear).getCbsaTable());

    if (matchingData.isPresent()) {
      final List<CbsaTableEntry> entries =
          matchingData.map(Map::entrySet).stream()
              .flatMap(s -> s.stream().map(Entry::getValue))
              .map(Map::entrySet)
              .flatMap(Collection::stream)
              .map(Entry::getValue)
              .sorted(
                  Comparator.comparing(CbsaTableEntry::getCbsa)
                      .thenComparing(CbsaTableEntry::getEffectiveDate))
              .collect(Collectors.toList());
      addEntries(pricerYear, entries);

      try {
        addCsv(pricerYear, csvMapper.writer(csvSchema).writeValueAsString(entries));
      } catch (final JsonProcessingException jpe) {
        throw new IllegalStateException(
            "Failed to convert entries to CSV for year " + pricerYear, jpe);
      }
    }
  }

  @GET
  @Operation(
      summary = "Retrieves all wage index data for a given fiscal year.",
      description =
          "Retrieves all CBSA wage index data for all effective dates within the provided fiscal "
              + "year.",
      responses = {
        @ApiResponse(
            responseCode = "200",
            description = "The matching CBSA data",
            content = {
              @Content(
                  mediaType = MediaType.APPLICATION_JSON,
                  schema = @Schema(implementation = CbsaRetrievalResponse.class)),
              @Content(mediaType = MEDIA_TYPE_CSV)
            }),
        @ApiResponse(responseCode = "400", description = "Invalid request"),
        @ApiResponse(responseCode = "404", description = "Year not found"),
        @ApiResponse(responseCode = "406", description = "Unsupported media type requested"),
        @ApiResponse(responseCode = "500", description = "Data retrieval error")
      })
  @Path("{year}")
  @Produces({MediaType.APPLICATION_JSON, MEDIA_TYPE_CSV})
  @Timed
  public Response findAll(
      @Max(9999)
          @Min(2000)
          @NotNull
          @PathParam("year")
          @Parameter(description = "The fiscal year for which data will be retrieved.")
          int year,
      @HeaderParam(HttpHeaders.ACCEPT) String contentType) {
    return getPerYearEntries(year, contentType, CbsaRetrievalResponse::new);
  }

  @GET
  @Operation(
      summary = "Retrieve all matching CBSA data for a given year.",
      description =
          "Returns the data for a specific CBSA for matching effective dates within the provided "
              + "fiscal year.",
      responses = {
        @ApiResponse(
            responseCode = "200",
            description = "The matching CBSA data",
            content =
                @Content(
                    mediaType = MediaType.APPLICATION_JSON,
                    schema = @Schema(implementation = CbsaRetrievalResponse.class))),
        @ApiResponse(responseCode = "400", description = "Invalid request"),
        @ApiResponse(responseCode = "404", description = "Year or specified CBSA not found"),
        @ApiResponse(responseCode = "500", description = "Data retrieval error")
      })
  @Path("{year}/{cbsa}")
  @Produces(MediaType.APPLICATION_JSON)
  @Timed
  public Response findMatching(
      @Max(9999)
          @Min(2000)
          @NotNull
          @Parameter(
              description = "The fiscal year for which data will be retrieved.",
              required = true)
          @PathParam("year")
          int year,
      @NotNull
          @Parameter(description = "The CBSA for which data will be retrieved.", required = true)
          @PathParam("cbsa")
          @Pattern(regexp = "\\d{2}|[1-9]\\d{4}", message = "must be either 2 or 5 digits")
          String cbsa,
      @Parameter(
              description =
                  "Filters data so that only the CBSA data effective as of the provided date will "
                      + "be returned; if omitted, all values for the year will be returned.",
              example = "2020-04-01",
              schema = @Schema(implementation = LocalDate.class))
          @QueryParam("effectiveDate")
          LocalDateParam effectiveDate) {
    return findMatchingEntries(
        () -> {
          if (null != effectiveDate) {
            final Optional<CbsaTableEntry> matchedEntry =
                Optional.ofNullable(DataTables.forYear(year))
                    .map(dt -> dt.getCbsaWageIndex(cbsa, effectiveDate.get()));

            return matchedEntry.map(Collections::singletonList).orElseGet(ArrayList::new);
          } else {
            final Optional<Collection<CbsaTableEntry>> matchingData =
                Optional.ofNullable(DataTables.forYear(year))
                    .map(DataTables::getCbsaTable)
                    .map(ct -> ct.get(cbsa))
                    .map(Map::values);

            // Check if year is present
            return matchingData.map(ArrayList::new).orElseGet(ArrayList::new);
          }
        },
        CbsaRetrievalResponse::new);
  }
}
