package gov.cms.fiss.pricers.ipps.resources;

import com.codahale.metrics.annotation.Timed;
import gov.cms.fiss.pricers.common.api.DateValidationException;
import gov.cms.fiss.pricers.common.api.InternalPricerException;
import gov.cms.fiss.pricers.common.api.YearNotImplementedException;
import gov.cms.fiss.pricers.common.application.errors.InputValidationError;
import gov.cms.fiss.pricers.common.application.errors.InternalPricerError;
import gov.cms.fiss.pricers.ipps.api.IppsInput;
import gov.cms.fiss.pricers.ipps.api.IppsOutput;
import gov.cms.fiss.pricers.ipps.core.IppsPricerDispatch;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

@Consumes(MediaType.APPLICATION_JSON)
@Path("/price-claim")
@Produces(MediaType.APPLICATION_JSON)
public class IppsPricerResource {
  private final IppsPricerDispatch dispatch;

  public IppsPricerResource(IppsPricerDispatch dispatch) {
    this.dispatch = dispatch;
  }

  @POST
  @Timed
  @Operation(
      summary = "Prospective payment record for the given IPPS bill",
      description =
          "Returns the Prospective Payment Record of the passed in IPPS Claim. This record "
              + "includes details on how it was calculated as well as potential adjustments made.",
      responses = {
        @ApiResponse(
            responseCode = "200",
            description = "The priced IPPS claim",
            content =
                @Content(
                    mediaType = MediaType.APPLICATION_JSON,
                    schema = @Schema(implementation = IppsOutput.class))),
        @ApiResponse(
            responseCode = "400",
            description =
                "Error: Bad Request.\n This status code covers requests that failed because of a "
                    + "problem with the syntax of the user's request:\n\n"
                    + "1) Request isn't valid JSON - The JSON body itself couldn't be parsed\n",
            content = @Content(mediaType = MediaType.APPLICATION_JSON)),
        @ApiResponse(
            responseCode = "422",
            description =
                "Error: Unprocessable Entity.\n This status code covers requests that failed "
                    + "because of a problem with the values in the user's request:\n\n"
                    + "1) Request doesn't match schema - The request fails to pass a required "
                    + "field, passes the wrong datatype for a field, or passes an invalid enum "
                    + "description, or some other similar problem.\n"
                    + "2) Request is not semantically valid - The request contains conflicting "
                    + "values, for example if exactly one of fieldA or fieldB is accepted, but the "
                    + "caller passes values for both fields.\n"
                    + "3) Request values are out of bounds - The request passes a numeric value "
                    + "that is too large or too small, passes an array with too many elements, or "
                    + "a string that is empty or has invalid characters.\n"
                    + "4) Request contains invalid dates - The request passes in date values that "
                    + "cannot be parsed correctly due to not being in the format of yyyyMMDD, was "
                    + "not a numerical value, or was greater than the allowed number of digits "
                    + "(8).\n"
                    + "5) A pricer year was provided that is not implemented",
            content =
                @Content(
                    mediaType = MediaType.APPLICATION_JSON,
                    schema = @Schema(implementation = InputValidationError.class))),
        @ApiResponse(
            responseCode = "500",
            description =
                "Error: Internal pricer error. Please save the request data and send it to ffsm-team at ffsm@bellese.io if you see this message.",
            content =
                @Content(
                    mediaType = MediaType.APPLICATION_JSON,
                    schema = @Schema(implementation = InternalPricerError.class)))
      })
  public IppsOutput price(
      @NotNull
          @Valid
          @Parameter(
              description = "IPPS claim to be priced",
              required = true,
              schema = @Schema(implementation = IppsInput.class))
          IppsInput input)
      throws YearNotImplementedException, InternalPricerException, DateValidationException {
    input.validate();

    return dispatch.process(input);
  }
}
