package gov.cms.hh.v04123;

import gov.cms.hh.claim.ClaimIF;
import gov.cms.hh.data.files.TableNames_EN;
import gov.cms.hh.data.meta.enumer.EditId_EN;
import gov.cms.hh.data.meta.enumer.EditType_EN;
import gov.cms.hh.diagnosis.ClinicalGroupIF;
import gov.cms.hh.diagnosis.CodeFirstCondition;
import gov.cms.hh.diagnosis.CodeFirstConditionIF;
import gov.cms.hh.diagnosis.DiagnosisIF;
import gov.cms.hh.grouper.DataManagerIF;
import gov.cms.hh.logic.specification.condition.ConditionChecker;
import gov.cms.hh.logic.validation.Edit;
import gov.cms.hh.reference.objects.CodeFirstConditionObject;
import gov.cms.hh.v04023.HippsPosition2_v04023;
import java.text.ParseException;
import java.util.List;
import java.util.logging.Level;

/**
 *
 * @author CMS
 */
public class HippsPosition2_v04123 extends HippsPosition2_v04023 {

    /**
     *
     * @param claim
     * @param dataManager
     */
    public HippsPosition2_v04123(ClaimIF claim, DataManagerIF dataManager) {
        super(claim, dataManager);
    }

    /**
     *
     * @param label
     * @param pdx
     * @return
     */
    @Override
    protected boolean checkAsPdx(String label, DiagnosisIF pdx) {
        /* Return Codes in use */
        final int GRC_INVALID_PDX = 3;
        final int GRC_UNGROUPABLE_PDX = 5;

        /* Validity Flags in use */
        final int GVF_UNACCEPTABLE_PDX = 1;
        final int GVF_MANIFESTATION_PDX = 2;
        final int GVF_UNSPECIFIED_PDX = 3;
        final int GVF_ECOI_PDX = 4;
        final int GVF_CODE_FIRST_PDX = 5;
        final int GVF_INVALID_PDX = 6;
        final int GVF_BLANK_PDX = 7;

        boolean retFlag = true;

        if (!pdx.getEditCollection().hasEditId(EditId_EN.VALUE_INVALID)) {
            // check Clinical Group
            ClinicalGroupIF cg = pdx.getClinicalGroup();
            if (cg == null || cg.getValue().isEmpty() || cg.getValue().equalsIgnoreCase("na")) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " has no Clinical Group assigned, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
            // check if manifestation
            if (pdx.isManifestation()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is manifestation DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_INVALID_PDX);
                // add validity flag
                addValidityFlag(GVF_MANIFESTATION_PDX);
                retFlag = false;
            }
            // check if ECOI
            if (pdx.isECOI()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is ECOI DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_INVALID_PDX);
                // add validity flag
                addValidityFlag(GVF_ECOI_PDX);
                retFlag = false;
            }
            // check if Unacceptable PDX
            if (pdx.isUnacceptablePdx()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is unacceptable as PDX DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_INVALID_PDX);
                // add validity flag
                addValidityFlag(GVF_UNACCEPTABLE_PDX);
                retFlag = false;
            }
            // check if Unspecified PDX
            if (pdx.isUnspecifiedPdx()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is unspecified as PDX DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_INVALID_PDX);
                // add validity flag
                addValidityFlag(GVF_UNSPECIFIED_PDX);
                retFlag = false;
            }
            // Code First 
            if (pdx.isCodeFirst()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_CODE_FIRST,
                                label + " " + pdx + " has Code First Condition, "
                                + EditId_EN.VALUE_CODE_FIRST.getDescription(), EditType_EN.CONSISTENCY, Level.INFO)
                );
                if (checkCodeFirstCondition(pdx)) {
                    // add validity flag only
                    addValidityFlag(GVF_CODE_FIRST_PDX);
                }
            }
        } else {
            // Report Invalid & Blank PDX separately
            addReturnCode(GRC_INVALID_PDX);
            if (pdx.getValue().trim().isEmpty()) {
                addValidityFlag(GVF_BLANK_PDX);
            } else {
                addValidityFlag(GVF_INVALID_PDX);
            }
            retFlag = false;
        }

        return retFlag;

    }

    private boolean checkCodeFirstCondition(DiagnosisIF pdx) {
        // 1. get condition reference
        String range = pdx.getCodeFirstCondition().getValue();
        // 2. build specs object
        ConditionChecker cfc = new ConditionChecker(range);
        // 3. check all SDXs against specs object and label all possible SDXs which meet CF
        List<DiagnosisIF> sdxList = getClaim().getSecondaryDiagnoses().getCollection();
        for (DiagnosisIF dx : sdxList) {
            if (dx.isValid()) {
                if (cfc.isCodeFirstConditionSatisfied(dx.getValue())) {
                    String edit = "SDX = %s, " + EditId_EN.VALUE_CODE_FIRST.getDescription();
                    dx.addEdit(
                        new Edit(EditId_EN.VALUE_CODE_FIRST, String.format(edit, dx),
                                EditType_EN.SPECIFICATION, Level.INFO)
                    );
                    // Add logic info edits for Detail Report
                    // SDX Code First info
                    addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, String.format(edit,
                            dx.getValue() + " (" + dx.getDescription() + ")"));
                    // return right after first DX found (per CAs)
                    return true;
                }
            }
        }
        return false;
    }

    private CodeFirstConditionObject getCodeFirstConditionReference(int id) {
        for (Object obj : getDataManager().getDataMap().get(TableNames_EN.CodeFirst_Conditions).getData()) {
            if (Integer.parseInt(((CodeFirstConditionObject) obj).getId()) == id) {
                return (CodeFirstConditionObject) obj;
            }
        }
        return null;
    }

    /**
     *
     * @param value
     * @return
     * @throws ParseException
     */
    @Override
    protected DiagnosisIF getDiagnosisReference(String value) throws ParseException {
        DiagnosisIF dx = super.getDiagnosisReference(value);
        if(dx != null) { // for valid DX only
            if(dx.isCodeFirst()) {
                CodeFirstConditionIF cfc = new CodeFirstCondition(getCodeFirstConditionReference(dx.getCodeFistConditionId()));
                dx.setCodeFirstCondition(cfc);
            }
        }
        return dx;
    }
    
}
