/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.application;

import gov.cms.hh.grouper.GrouperFactory;
import gov.cms.hh.data.exchange.ClaimContainerIF;
import gov.cms.hh.data.meta.MetaData;
import gov.cms.hh.data.meta.enumer.CliReturn_EN;
import gov.cms.hh.grouper.GrouperFactoryIF;
import gov.cms.hh.grouper.GrouperIF;
import java.io.File;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public final class HHGrouper {

    private static final Logger LOG = Logger.getLogger(HHGrouper.class.getName());
    private static final String CMD_LINE_DESIRED_FORMAT_MSG = "Required argument: <Input File/Record>, "
            + "optional: <Full output record>: -f, (arguments are positional)";
    private static GrouperFactoryIF GROUPER_FACTORY = null;
    private static final String FILE_OUT_SUFFIX = "_OUT";
    private static final String TEST_OUTPUT = "AA.A.AABBBBBCCDD";
    private static File inputFile;
    private static File outputFile;
    private static boolean fullRecord;
    private static boolean loadDescription;

    /*
     *   PC API starts here
     */
    /**
     * Provide empty claim object to be populated for grouping
     *
     * @param claimDate
     * @return empty claim object
     */
    public final static ClaimContainerIF getEmptyClaim(String claimDate) {
        return getGrouperFactory().getEmptyClaim(claimDate);
    }

    /**
     * Check if claim date can be processed by grouper
     *
     * @param claimDate
     * @return claim from date check
     */
    public final static boolean isClaimDateValid(String claimDate) {
        return getGrouperFactory().isClaimDateValid(claimDate);
    }

    /**
     * Get Diagnosis description based on claim date
     *
     * @param claimDate
     * @param diagValue
     * @return
     */
    public final static String getDiagnosisDescription(String claimDate, String diagValue) {
        return getGrouperFactory().getDiagnosisDescription(claimDate, diagValue);
    }

    /**
     * Get Build Info
     *
     * @return
     */
    public final static String getBuildInfo() {
        return getGrouperFactory().getBuildInfo();
    }

    /**
     * Group claim
     *
     * @param claim
     */
    public final static void group(ClaimContainerIF claim) {
        getGrouperFactory().group(claim);
    }

    // for content viewers
    /**
     * Get available grouper versions
     *
     * @return
     */
    public final static List<GrouperIF> getAvailableVersions() {
        return getGrouperFactory().getAvailableVersions();
    }

    // for content viewers
    /**
     * Get default grouper version
     *
     * @return
     */
    public final static GrouperIF getDefaultGrouper() {
        return getGrouperFactory().getDefaultGrouper();
    }

    /*
     *   Batch and MF API starts here
     */
    /**
     * Produces output file with output portion only
     *
     * @param input
     * @param output
     */
    public final static void group(File input, File output) {
        getGrouperFactory().group(input, output);
    }

    /**
     * Could produce output file with input + output portions if 'fullRecord' =
     * true
     *
     * @param input
     * @param output
     * @param fullRecord
     */
    public final static void group(File input, File output, boolean fullRecord) {
        getGrouperFactory().group(input, output, fullRecord);
    }

    /**
     * Grouper "empty" test call
     *
     * @param claimInput
     * @return
     */
    public final static String groupTest(String claimInput) {
        return TEST_OUTPUT;
    }

    /**
     * Group claim in fixed format
     *
     * @param claimInput
     * @return
     */
    public final static String group(String claimInput) {
        return getGrouperFactory().group(claimInput);
    }

    /**
     * Group claim in fixed format with full record output
     *
     * @param claimInput
     * @param fullRecord
     * @return
     */
    public final static String group(String claimInput, boolean fullRecord) {
        return getGrouperFactory().group(claimInput, fullRecord);
    }

    /**
     * Batch mode
     *
     * @param args
     */
    public static void main(String[] args) {
        LOG.info(getBuildInfo());
        CliReturn_EN ret = processCmdLineArgs(args);
        if (ret == CliReturn_EN.RUN_NOTHING) {
            return;
        }
        if (ret == CliReturn_EN.RUN_FILE) {
            group(inputFile, outputFile, fullRecord);
        }
        if (ret == CliReturn_EN.RUN_RECORD) {
            String input = inputFile.getName();
            String output = group(input, fullRecord);
            System.out.print(output);
        }
    }

    private static CliReturn_EN processCmdLineArgs(final String args[]) {
        if (args.length > 0) {
            for (int i = 0; i < args.length; i++) {
                String arg = args[i];
                switch (i) {
                    case 0:
                        inputFile = new File(arg);
                        break;
                    case 1:
                        if (arg.equalsIgnoreCase(MetaData.FLAG_FULL_RECORD)) {
                            fullRecord = true;
                        }
                        break;
                    default:
                        LOG.log(Level.WARNING, "Unknown parameter: {0}", args[i]);
                }
            }
            if (inputFile.exists()) {
                outputFile = getOutputFile(inputFile);
                LOG.log(Level.INFO, "Grouping: input file=<{0}>, output file=<{1}>, full record format=<{2}>",
                        new Object[]{inputFile.getPath(), outputFile.getPath(), fullRecord});
                return CliReturn_EN.RUN_FILE;
            } else {
                String input = inputFile.getName();
                if (input.trim().isEmpty()) {
                    LOG.log(Level.SEVERE, "Input file/record is empty");
                } else {
                    LOG.log(Level.INFO, "Could not find input file:\n''{0}''\nWill try "
                            + "to process it as a record.", input);
                    return CliReturn_EN.RUN_RECORD;
                }
            }
        } else {
            LOG.severe(CMD_LINE_DESIRED_FORMAT_MSG);
        }
        return CliReturn_EN.RUN_NOTHING;
    }

    private static File getOutputFile(File file) {
        String fileName = file.getPath();
        String fileOut;
        int pos = fileName.lastIndexOf('.');
        if (pos > 0 && pos < (fileName.length() - 1)) { // make sure '.' is not the first or last character.
            fileOut = fileName.substring(0, pos) + FILE_OUT_SUFFIX + fileName.substring(pos);
        } else {
            fileOut = fileName + FILE_OUT_SUFFIX;
        }
        return new File(fileOut);
    }

    /**
     * @return the GROUPER_FACTORY
     */
    private static GrouperFactoryIF getGrouperFactory() {
        // lazy instantiation
        // to let run test calls first
        if (GROUPER_FACTORY == null) {
            GROUPER_FACTORY = new GrouperFactory(loadDescription);
        }
        return GROUPER_FACTORY;
    }

    /**
     * @return the loadDescription
     */
    public static boolean isLoadDescription() {
        return loadDescription;
    }

    /**
     * @param aLoadDescription the loadDescription to set
     */
    public static void setLoadDescription(boolean aLoadDescription) {
        loadDescription = aLoadDescription;
    }
}
