/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.reference.objects;

import gov.cms.hh.data.exchange.ClaimContainerIF;
import gov.cms.hh.data.exchange.DxContainer;
import gov.cms.hh.data.exchange.DxContainerIF;
import gov.cms.hh.data.exchange.HippsContainerIF;
import gov.cms.hh.data.exchange.ReturnCodeContainerIF;
import gov.cms.hh.data.exchange.ValidityFlagContainerIF;
import gov.cms.hh.data.loader.PositionAccessorIF;
import gov.cms.hh.data.meta.enumer.Claim_EN;
import gov.cms.hh.data.meta.enumer.Oasis_EN;
import gov.cms.hh.logic.validation.EditCollectionIF;
import gov.cms.hh.xml.data.claim.Claim;
import gov.cms.hh.xml.data.claim.ClinicalFields;
import gov.cms.hh.xml.data.claim.Diagnoses;
import gov.cms.hh.xml.data.claim.Dx;
import gov.cms.hh.xml.data.claim.FunctionalField;
import gov.cms.hh.xml.data.claim.FunctionalFields;
import gov.cms.hh.xml.data.claim.GrouperReturnCode;
import gov.cms.hh.xml.data.claim.HippsCode;
import gov.cms.hh.xml.data.claim.Input;
import gov.cms.hh.xml.data.claim.Options;
import gov.cms.hh.xml.data.claim.Output;
import gov.cms.hh.xml.data.claim.Principal;
import gov.cms.hh.xml.data.claim.Secondary;
import gov.cms.hh.xml.data.claim.Subfield;
import gov.cms.hh.xml.data.claim.ValidityFlag;
import gov.cms.hh.xml.data.claim.VersionUsed;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 *
 * @author CMS
 */
public class ClaimObject extends Claim implements PositionAccessorIF, ClaimContainerIF {

    /**
     * Constructor
     */
    public ClaimObject() {
        // Create all inner objects upfront
        init();
    }

    private void init() {
        setOptions(new Options());
        setInput(new Input());
        getInput().setClinicalFields(new ClinicalFields());
        getInput().getClinicalFields().setDiagnoses(new Diagnoses());
        getInput().getClinicalFields().getDiagnoses().setPrincipal(new Principal());
        getInput().getClinicalFields().getDiagnoses().setSecondary(new Secondary());
        getInput().setFunctionalFields(new FunctionalFields());
        setOutput(new Output());
        getOutput().setVersionUsed(new VersionUsed());
        getOutput().setHippsCode(new HippsCode());
        getOutput().setValidityFlag(new ValidityFlag());
        getOutput().setGrouperReturnCode(new GrouperReturnCode());
    }

    /**
     *
     * @return
     */
    @Override
    public String toString() {

        String nl = "\n";
        return "ClaimObject{" + nl
                + "ID='" + getId() + "'" + nl
                + "Description='" + (getDescription() == null ? "" : getDescription()) + "'" + nl
                + "From Date='" + getInput().getClinicalFields().getFromDate() + "'" + nl
                + "Through Date='" + getInput().getClinicalFields().getThroughDate() + "'" + nl
                + "Period Timing='" + getInput().getClinicalFields().getPeriodTiming() + "'" + nl
                + "Referral Source='" + getInput().getClinicalFields().getReferralSource() + "'" + nl
                + formatDx(getInput().getClinicalFields().getDiagnoses().getPrincipal().getDx(), "P", nl)
                + formatDx(getInput().getClinicalFields().getDiagnoses().getSecondary().getDx(), "S", nl)
                + formatOasis(getInput().getFunctionalFields().getFunctionalField(), nl)
                + "Version Used='" + formatVersion(getOutput().getVersionUsed()) + "'" + nl
                + "HIPPS Code='" + formatHippsCode(getOutput().getHippsCode()) + "'" + nl
                + "Validity Flag='" + getOutput().getValidityFlag().getValue() + "'" + nl
                + "Grouper Return Code='" + getOutput().getGrouperReturnCode().getValue() + "'" + nl
                + "ActionFlag='" + getOptions().getActionFlag() + "'" + nl
                + "}";
    }

    /**
     *
     * @param position
     * @return
     */
    @Override
    public String getByPosition(int position) {
        
        if (position == Claim_EN.ID.getItemNumberStart()) {
            return getId();
        }
        
        if (position == Claim_EN.FROM_DATE.getItemNumberStart()) {
            return getInput().getClinicalFields().getFromDate();
        }
        
        if (position == Claim_EN.PERIOD_TIMING.getItemNumberStart()) {
            return getInput().getClinicalFields().getPeriodTiming();
        }
        
        if (position == Claim_EN.REFERRAL_SOURCE.getItemNumberStart()) {
            return getInput().getClinicalFields().getReferralSource();
        }

        if (position == Claim_EN.PDX.getItemNumberStart()) {
            Dx dx = getInput().getClinicalFields().getDiagnoses().getPrincipal().getDx();
            String retDx = dx.getCode() + dx.getPoa();
            return retDx;
        }
        
        if (position >= Claim_EN.SDX.getItemNumberStart() && position <= Claim_EN.SDX.getItemNumberEnd()) {
            return getInput().getClinicalFields().getDiagnoses().getSecondary().getDx().get(position - 5).getCode();
        }
        
        // Filler Fields from 29 to 33
        
        // M1033
        if (position >= Claim_EN.M1033.getItemNumberStart() && position <= Claim_EN.M1033.getItemNumberEnd()) {
            getInput().getFunctionalFields().getFunctionalField().get(0).getSubfield().get(position - 34).getValue();
        }
        
        // M18XX fields
        if (position >= Claim_EN.M18XX.getItemNumberStart() && position <= Claim_EN.M18XX.getItemNumberEnd()) {
            getInput().getFunctionalFields().getFunctionalField().get(position - 44).getSubfield().get(0).getValue();
        }
        
        if (position == Claim_EN.THROUGH_DATE.getItemNumberStart()) {
            return getInput().getClinicalFields().getThroughDate();
        }        

        return null; // Exception???
    }

    /**
     *
     * @param position
     * @param value
     * @return
     */
    @Override
    public boolean setByPosition(int position, String value) {

        // Claim ID
        if (position == Claim_EN.ID.getItemNumberStart()) {
            setId(value);
        }

        // Claim From Date
        if (position == Claim_EN.FROM_DATE.getItemNumberStart()) {
            getInput().getClinicalFields().setFromDate(value);
        }

        // Claim Period Timing
        if (position == Claim_EN.PERIOD_TIMING.getItemNumberStart()) {
            getInput().getClinicalFields().setPeriodTiming(value);
        }

        // Claim Referral Source
        if (position == Claim_EN.REFERRAL_SOURCE.getItemNumberStart()) {
            getInput().getClinicalFields().setReferralSource(value);
        }

        if (position == Claim_EN.PDX.getItemNumberStart()) {
            Dx dx = new Dx();
            dx.setCode(value.substring(0, value.length() - 1));   // DX
            dx.setPoa(value.substring(value.length() - 1));       // POA
            dx.setOccurrence("0");
            getInput().getClinicalFields().getDiagnoses().getPrincipal().setDx(dx);
        }

        if (position >= Claim_EN.SDX.getItemNumberStart() && position <= Claim_EN.SDX.getItemNumberEnd()) {
            List<Dx> dxList = getInput().getClinicalFields().getDiagnoses().getSecondary().getDx();
            Dx dx = new Dx();
            dx.setCode(value.substring(0, value.length() - 1));   // DX
            dx.setPoa(value.substring(value.length() - 1));       // POA
            dx.setOccurrence("" + (dxList.size() + 1));
            dxList.add(dx);
        }

        // M1033
        if (position >= Claim_EN.M1033.getItemNumberStart() && position <= Claim_EN.M1033.getItemNumberEnd()) {
            FunctionalField ff = null;
            if (position == Claim_EN.M1033.getItemNumberStart()) {
                ff = new FunctionalField();
                String name = Oasis_EN.values()[0].name().substring(0, 5);
                ff.setFieldName(name);
                getInput().getFunctionalFields().getFunctionalField().add(ff);
            } else {
                ff = getInput().getFunctionalFields().getFunctionalField().get(0);
            }
            Subfield sf = new Subfield();
            sf.setName(Oasis_EN.values()[position - Claim_EN.M1033.getItemNumberStart()].name());
            sf.setValue(value);
            ff.getSubfield().add(sf);
        }

        // M18XX fields
        if (position >= Claim_EN.M18XX.getItemNumberStart() && position <= Claim_EN.M18XX.getItemNumberEnd()) {
            FunctionalField ff = new FunctionalField();
            Subfield sf = new Subfield();
            String name = Oasis_EN.values()[position - (Claim_EN.M18XX.getItemNumberStart() - 10)].name();
            ff.setFieldName(name.substring(0, 5));
            sf.setName(name);
            sf.setValue(value);
            ff.getSubfield().add(sf);
            getInput().getFunctionalFields().getFunctionalField().add(ff);
        }

        // position = 51 is filler
        
        // Claim Through Date
        if (position == Claim_EN.THROUGH_DATE.getItemNumberStart()) {
            getInput().getClinicalFields().setThroughDate(value);
        }

        // position = 53 is filler
        
        // Options
        // Action Flag
        if (position == Claim_EN.ACTION_FLAG.getItemNumberStart()) {
            getOptions().setActionFlag(value);
        }
        // Output
        if (position == Claim_EN.VERSION.getItemNumberStart()) {
            VersionUsed vu = getOutput().getVersionUsed();
            if (value.contains(".")) {
                vu.setVersion(value.substring(0, 2));
                vu.setRelease(value.substring(3, 4));
                vu.setYear(value.substring(5));
            } else {
                vu.setVersion(value);
                vu.setRelease("");
                vu.setYear("");
            }
        }
        if (position == Claim_EN.HIPPS.getItemNumberStart()) {
            HippsCode hc = getOutput().getHippsCode();
            hc.setEpisodeTiming(value.substring(0, 1));
            hc.setClinicalGroup(value.substring(1, 2));
            hc.setFunctionalLevel(value.substring(2, 3));
            hc.setComorbidityPresent(value.substring(3, 4));
            hc.setFiller(value.substring(4, 5));

        }
        if (position == Claim_EN.VALIDITY_FLAG.getItemNumberStart()) {
            ValidityFlag vf = getOutput().getValidityFlag();
            vf.setValue(value);
        }
        if (position == Claim_EN.RETURN_CODE.getItemNumberStart()) {
            GrouperReturnCode grc = getOutput().getGrouperReturnCode();
            grc.setValue(value);
        }

        return false; // Exception???
    }

    private String formatDx(Dx dx, String pref, String del) {
        if(dx.getCode().trim().isEmpty()) {
            return null;
        }
        return pref + "DX='{'" + dx.getCode() + "','" + dx.getPoa() + "','" + dx.getOccurrence() + "'}" + del;
    }

    private String formatDx(List<Dx> dxList, String pref, String del) {
        // get rid of empty SDXs
        StringBuilder ret = new StringBuilder();
        dxList.stream().map((dx) -> formatDx(dx, pref, del)).filter((retStr) -> (retStr != null)).forEachOrdered((retStr) -> {
            ret.append(retStr);
        });
        return ret.toString();
    }

    private String formatOasis(List<FunctionalField> ffList, String del) {
        String ret = "";
        for (FunctionalField ff : ffList) {
            List<Subfield> sfList = ff.getSubfield();
            ret = sfList.stream().map((sf) -> "OASIS Field={" + sf.getName() + "='" + sf.getValue() + "'}" + del).reduce(ret, String::concat);
        }
        return ret;
    }

    private String formatVersion(VersionUsed vu) {
        String ret = "";
        ret = vu.getVersion() + "." + vu.getRelease() + "." + vu.getYear();
        return ret;
    }

    private String formatHippsCode(HippsCode hc) {
        String ret = "";
        ret = hc.getEpisodeTiming() + hc.getClinicalGroup() + hc.getFunctionalLevel() + hc.getComorbidityPresent() + hc.getFiller();
        return ret;
    }

    /**
     *
     * @return
     */
    public String tempGetOutput() {
        return formatVersion(getOutput().getVersionUsed())
                + formatHippsCode(getOutput().getHippsCode())
                + getOutput().getValidityFlag().getValue()
                + getOutput().getGrouperReturnCode().getValue();
    }

    /**
     *
     * @return
     */
    @Override
    public int getNumberOfFields() {
        return 57;
    }

    private boolean loadDescription;

    /**
     * @return the loadDescription
     */
    @Override
    public boolean isLoadDescription() {
        return loadDescription;
    }

    /**
     * @param loadDescription the loadDescription to set
     */
    @Override
    public void setLoadDescription(boolean loadDescription) {
        this.loadDescription = loadDescription;
    }

    /**
     *
     * @return
     */
    @Override
    public String getHippsCode() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getReturnCodeValue() {
        return getOutput().getGrouperReturnCode().getValue();
    }

    /**
     *
     * @return
     */
    @Override
    public String getReturnCodeDescription() {
        return getOutput().getGrouperReturnCode().getDescription();
    }

    /**
     *
     * @return
     */
    @Override
    public String getValidityFlagValue() {
        return getOutput().getValidityFlag().getValue();
    }

    /**
     *
     * @return
     */
    @Override
    public String getValidityFlagDescription() {
        return getOutput().getValidityFlag().getDescription();
    }

    /**
     *
     * @return
     */
    @Override
    public String getVersionUsed() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param ambulation
     */
    @Override
    public void setAmbulation(String ambulation) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param bathing
     */
    @Override
    public void setBathing(String bathing) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getClaimId() {
        return getId();
    }

    /**
     *
     * @param claimId
     */
    @Override
    public void setClaimId(String claimId) {
        setId(claimId);
    }

    /**
     *
     * @param dressLower
     */
    @Override
    public void setDressLower(String dressLower) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param dressUpper
     */
    @Override
    public void setDressUpper(String dressUpper) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param fromDate
     */
    @Override
    public void setFromDate(String fromDate) {
        getInput().getClinicalFields().setFromDate(fromDate);
    }

    /**
     *
     * @param throughDate
     */
    @Override
    public void setThroughDate(String throughDate) {
        getInput().getClinicalFields().setThroughDate(throughDate);
    }
    
    /**
     *
     * @param grooming
     */
    @Override
    public void setGrooming(String grooming) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskCompliance
     */
    @Override
    public void setHospRiskCompliance(String hospRiskCompliance) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskFiveMoreMeds
     */
    @Override
    public void setHospRiskFiveMoreMeds(String hospRiskFiveMoreMeds) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskHistoryFalls
     */
    @Override
    public void setHospRiskHistoryFalls(String hospRiskHistoryFalls) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskMentalBehavDecl
     */
    @Override
    public void setHospRiskMentalBehavDecl(String hospRiskMentalBehavDecl) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskMultiEdVisit
     */
    @Override
    public void setHospRiskMultiEdVisit(String hospRiskMultiEdVisit) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskMultiHospital
     */
    @Override
    public void setHospRiskMultiHospital(String hospRiskMultiHospital) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskNoneAbove
     */
    @Override
    public void setHospRiskNoneAbove(String hospRiskNoneAbove) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskOtherRisk
     */
    @Override
    public void setHospRiskOtherRisk(String hospRiskOtherRisk) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskExhaustion
     */
    @Override
    public void setHospRiskExhaustion(String hospRiskExhaustion) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hospRiskWeightLoss
     */
    @Override
    public void setHospRiskWeightLoss(String hospRiskWeightLoss) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param pdx
     * @param poa
     */
    @Override
    public void setPdx(String pdx, String poa) {
        Dx dx = new Dx();
        dx.setCode(pdx);   // DX
        dx.setPoa(poa);   // POA
        dx.setOccurrence("0");
        getInput().getClinicalFields().getDiagnoses().getPrincipal().setDx(dx);
    }

    /**
     *
     * @param sdx
     * @param poa
     */
    @Override
    public void addSdx(String sdx, String poa) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param periodTiming
     */
    @Override
    public void setPeriodTiming(String periodTiming) {
        getInput().getClinicalFields().setPeriodTiming(periodTiming);
    }

    /**
     *
     * @param referralSource
     */
    @Override
    public void setReferralSource(String referralSource) {
        getInput().getClinicalFields().setReferralSource(referralSource);
    }

    /**
     *
     * @param toileting
     */
    @Override
    public void setToileting(String toileting) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param transferring
     */
    @Override
    public void setTransferring(String transferring) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getAmbulation() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getBathing() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getDressLower() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getDressUpper() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getFromDate() {
        return getInput().getClinicalFields().getFromDate();
    }

    /**
     *
     * @return
     */
    @Override
    public String getThroughDate() {
        return getInput().getClinicalFields().getThroughDate();
    }

    /**
     *
     * @return
     */
    @Override
    public String getGrooming() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskCompliance() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskFiveMoreMeds() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskHistoryFalls() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskMentalBehavDecl() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskMultiEdVisit() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskMultiHospital() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskNoneAbove() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskOtherRisk() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskExhaustion() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getHospRiskWeightLoss() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public DxContainerIF getPdx() { // refactoring (close coupling) later?
        Dx dx = getInput().getClinicalFields().getDiagnoses().getPrincipal().getDx();
        DxContainerIF pdx = new DxContainer(dx.getCode(), dx.getPoa());
        return pdx;
    }

    /**
     *
     * @return
     */
    @Override
    public String getPeriodTiming() {
        return getInput().getClinicalFields().getPeriodTiming();
    }

    /**
     *
     * @return
     */
    @Override
    public String getReferralSource() {
        return getInput().getClinicalFields().getReferralSource();
    }

    /**
     *
     * @return
     */
    @Override
    public ReturnCodeContainerIF getReturnCode() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public List<DxContainerIF> getSdxList() {
        List<DxContainerIF> retList = null;
        List<Dx> dxList = getInput().getClinicalFields().getDiagnoses().getSecondary().getDx();
        for (Dx dx : dxList) {
            DxContainerIF diag = new DxContainer(dx.getCode(), dx.getPoa());
            if (retList == null) {
                retList = new ArrayList<>();
            }
            retList.add(diag);
        }
        return retList;
    }

    /**
     *
     * @return
     */
    @Override
    public String getToileting() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public String getTransferring() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public ValidityFlagContainerIF getValidityFlag() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param HippsCode
     */
    @Override
    public void setHippsCode(String HippsCode) {
        setByPosition(Claim_EN.HIPPS.getItemNumberStart(), HippsCode);
    }

    /**
     *
     * @param pdx
     */
    @Override
    public void setPdx(DxContainerIF pdx) {
        Dx dx = new Dx();
        dx.setCode(pdx.getDx());   // DX
        dx.setPoa(pdx.getPoa());   // POA
        dx.setOccurrence("0");
        getInput().getClinicalFields().getDiagnoses().getPrincipal().setDx(dx);
    }

    /**
     *
     * @param returnCode
     */
    @Override
    public void setReturnCode(ReturnCodeContainerIF returnCode) {
        getOutput().getGrouperReturnCode().setValue(returnCode.getValue());
        getOutput().getGrouperReturnCode().setDescription(returnCode.getDescription());
    }

    /**
     *
     * @param sdxList
     */
    @Override
    public void setSdxList(List<DxContainerIF> sdxList) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param validityFlag
     */
    @Override
    public void setValidityFlag(ValidityFlagContainerIF validityFlag) {
        getOutput().getValidityFlag().setValue(validityFlag.getValue());
        getOutput().getValidityFlag().setDescription(validityFlag.getDescription());
    }

    /**
     *
     * @param versionUsed
     */
    @Override
    public void setVersionUsed(String versionUsed) {
        setByPosition(Claim_EN.VERSION.getItemNumberStart(), versionUsed);
    }

    /**
     *
     * @return
     */
    @Override
    public Map<String, String> getFunctionalFields() {
        Map<String, String> retMap = new TreeMap<>();
        List<FunctionalField> ffList = getInput().getFunctionalFields().getFunctionalField();
        for (FunctionalField ff : ffList) {
            List<Subfield> sfList = ff.getSubfield();
            for (Subfield sf : sfList) {
                retMap.put(sf.getName(), sf.getValue());
            }
        }
        return retMap;
    }

    /**
     *
     * @return
     */
    @Override
    public EditCollectionIF getEdits() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param edits
     */
    @Override
    public void setEdits(EditCollectionIF edits) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @return
     */
    @Override
    public HippsContainerIF getHippsContainer() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     *
     * @param hippsContainerIF
     */
    @Override
    public void setHippsContainer(HippsContainerIF hippsContainerIF) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

}
