/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.data.loader;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.io.FileNotFoundException;
import java.util.stream.Stream;
import static gov.cms.hh.data.meta.MetaData.ENCODING_DEFAULT;

/**
 *
 * @author CMS
 * @param <T>
 */
public class DataLoader<T> implements DataLoaderIF {

    private static final Logger LOG = Logger.getLogger(DataLoader.class.getName());

    /**
     *
     * @param clazz
     */
    public DataLoader(Class<T> clazz) {
        this(clazz, new LineParser(), false);
    }
    
    /**
     *
     * @param clazz
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, boolean loadDescriptions) {
        this(clazz, new LineParser(), loadDescriptions);
    }

    /**
     *
     * @param clazz
     * @param recordLayout
     */
    public DataLoader(Class<T> clazz, RecordLayoutIF recordLayout) {
        this(clazz, recordLayout, false);
    }
    
    /**
     *
     * @param clazz
     * @param recordLayout
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, RecordLayoutIF recordLayout, boolean loadDescriptions) {
        this(clazz, new LineParser(recordLayout.getLayout()), loadDescriptions);
    }

    /**
     *
     * @param clazz
     * @param lineParser
     */
    public DataLoader(Class<T> clazz, LineParserIF lineParser) {
        this(clazz, lineParser, false);
    }
    
    /**
     *
     * @param clazz
     * @param lineParser
     * @param loadDescriptions
     */
    public DataLoader(Class<T> clazz, LineParserIF lineParser, boolean loadDescriptions) {
        this.mapTo = (String line) -> {
            List<String> tokens = lineParser.parseLine(line);
            PositionAccessorIF data;
            try {
                T t = clazz.newInstance();
                data = (PositionAccessorIF) t;
                data.setLoadDescription(loadDescriptions);
                for (int i = 0; i < tokens.size(); i++) {
                    data.setByPosition(i, tokens.get(i).intern());
                }
                return t;
            } catch (InstantiationException | IllegalAccessException ex) {
                Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
            }
            return null;
        };
    }

    /**
     *
     * @param inputFile
     * @param withHeaders
     * @return
     */
    public List<T> loadFile(InputStream inputFile, boolean withHeaders) {
        List<T> inputList = new ArrayList();
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputFile, ENCODING_DEFAULT));
            // skip the header of the tsv
            if (withHeaders) {
                inputList = br.lines().map(mapTo).collect(Collectors.toList());
            } else {
                inputList = br.lines().skip(1).map(mapTo).collect(Collectors.toList());
            }
            br.close();
        } catch (IOException ex) {
            Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
        }
        return inputList;

    }

    /**
     *
     * @param inputFile
     * @param withHeaders
     * @return
     */
    @Override
    public List<T> loadFile(String inputFile, boolean withHeaders) {
        File inputF = new File(inputFile);
        InputStream inputFS = null;
        try {
            inputFS = new FileInputStream(inputF);
        } catch (FileNotFoundException ex) {
            Logger.getLogger(DataLoader.class.getName()).log(Level.SEVERE, null, ex);
        }
        return loadFile(inputFS, withHeaders);
    }

    private final Function<String, T> mapTo;

    /**
     *
     * @param fileName
     * @return
     */
    @Override
    public List<T> loadFile(String fileName) {
        return loadFile(fileName, false);
    }

    /**
     *
     * @param inputFile
     * @return
     */
    public List<T> loadFile(InputStream inputFile) {
        return loadFile(inputFile, false);
    }

    /**
     *
     * @param s
     * @return
     */
    @Override
    public T load(String s) {
        return Stream.of(s).map(mapTo).collect(Collectors.toList()).get(0);
    }

}
