/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.grouper;

import gov.cms.hh.data.files.FileProperties;
import gov.cms.hh.data.files.TableNames_EN;
import gov.cms.hh.data.exchange.ClaimContainerIF;
import gov.cms.hh.data.loader.DataLoader;
import gov.cms.hh.reference.objects.ClaimObject;
import gov.cms.hh.data.exchange.ReturnCodeContainer;
import gov.cms.hh.data.exchange.ValidityFlagContainer;
import gov.cms.hh.data.exchange.ValidityFlagContainerIF;
import static gov.cms.hh.data.meta.MetaData.ENCODING_DEFAULT;
import gov.cms.hh.data.meta.enumer.ActionFlag_EN;
import gov.cms.hh.grouper.utility.Utility;
import gov.cms.hh.logic.specification.IsValidFromDate;
import gov.cms.hh.reference.objects.DiagnosisObject;
import gov.cms.hh.reference.objects.GrouperVersionObject;
import gov.cms.hh.reference.objects.ReturnCodeObject;
import gov.cms.hh.reference.objects.ValidityFlagObject;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Scanner;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 *
 * @author CMS
 */
public class GrouperFactory implements GrouperFactoryIF {

    private static final Logger LOG = Logger.getLogger(GrouperFactory.class.getName());

    private final String GROUP_VER_FILE
            = FileProperties.FILE_BASE + "/" + TableNames_EN.Grouper_Versions.getFileName();

    Map<String, GrouperIF> grouperList;

    /**
     * ctor
     *
     * @param loadDescription
     */
    public GrouperFactory(boolean loadDescription) {
        // Get Grouper_Versions from ROT
        grouperList = buildGrouperPool(GROUP_VER_FILE, loadDescription);
        if (grouperList == null) {
            LOG.log(Level.SEVERE, "Grouper factory instanting error");
        } else {
            LOG.log(Level.INFO, "Grouper factory instantiated with {0} version(s)", grouperList.size());
        }
    }

    /**
     *
     * @param claimDate
     * @return
     */
    @Override
    public ClaimContainerIF getEmptyClaim(String claimDate) {
        GrouperIF grouper = getGrouper(claimDate);
        if (grouper != null) {
            return grouper.getEmptyClaim();

        }
        return null;
    }

    /**
     *
     * @param claimDate
     * @param diagValue
     * @return
     */
    @Override
    public String getDiagnosisDescription(String claimDate, String diagValue) {
        GrouperIF grouper = getGrouper(claimDate);
        if (grouper != null) {
            DiagnosisObject dxObj = (DiagnosisObject) grouper.getAvailableTableContent(TableNames_EN.Diagnoses).getDataMap().get(diagValue);
            if (dxObj != null) {
                String desc = dxObj.getDescription();
                if (desc != null) {
                    return desc;
                } else {
                    return "Diagnosis description not found. Check grouper setup";
                }
            } else {
                return "Diagnosis code is not valid for claim dates";
            }
        } else {
            return "Grouper not found";
        }
    }

    /**
     *
     * @param claimIn
     */
    @Override
    public void group(ClaimContainerIF claimIn) {
        if (claimIn == null) {
            return;
        }
        if (claimIn.getFromDate() != null && isClaimDateValid(claimIn.getFromDate())) {
            GrouperIF grouper = getGrouper(claimIn.getFromDate());
            if (grouper != null) {
                grouper.group(claimIn);
            } else {
                // never should be here
                // grouper is incorrect
            }
        } else {
            setResultForIncorrectDate(claimIn);
        }
    }

    /**
     *
     * @param input
     * @param output
     */
    @Override
    public void group(File input, File output) {
        group(input, output, false);
    }

    /**
     *
     * @param input
     * @param output
     * @param fullRecord
     */
    @Override
    public void group(File input, File output, boolean fullRecord) {
        try {
            Scanner sc = new Scanner(input, ENCODING_DEFAULT);
            FileWriter fileWriter = new FileWriter(output);
            PrintWriter printWriter = new PrintWriter(fileWriter);
            while (sc.hasNextLine()) {
                printWriter.println(group(sc.nextLine(), fullRecord));
            }
            sc.close();
            printWriter.close();
        } catch (FileNotFoundException ex) {
            Logger.getLogger(GrouperFactory.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(GrouperFactory.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     *
     * @param claimInput
     * @return
     */
    @Override
    public String group(String claimInput) {
        return group(claimInput, false);
    }

    /**
     *
     * @param claimInput
     * @param fullRecord
     * @return
     */
    @Override
    public String group(String claimInput, boolean fullRecord) {
        // test only
//        return getEmptyResult().intern();

        /* */
        if (claimInput == null || claimInput.isEmpty()) {
            return getResultForIncorrectDate(claimInput);
        }
        // Usually "Claim From Date" is the second element of claim layout
        final int INDEX_CLAIM_ID = 0;
        final int INDEX_CLAIM_FROM_DATE = 1;
        // Attempt to get "Claim From Date" dynamiclly to select the appropriate grouper version
        // get 2nd layout element
        List<Integer> claimLayout = getDefaultGrouper().getClaimLayout();
        // parse Date From from input claim
        int claimDatePosition = claimLayout.get(INDEX_CLAIM_ID); // where claim id ends
        int claimDateLength = claimLayout.get(INDEX_CLAIM_FROM_DATE);
        // check input length
        if (claimInput.length() < claimDatePosition + claimDateLength) {
            return getResultForIncorrectDate(claimInput);
        }
        String fromDate = claimInput.substring(claimDatePosition, claimDatePosition + claimDateLength);
        // group
        GrouperIF grouper = getGrouper(fromDate);
        String retRes;
        // if grouper was not found
        // most likely due to invalid date
        int inputLength = -1;
        int outputLength = -1;
        if (grouper == null) {
            retRes = getResultForIncorrectDate(claimInput);
            grouper = getDefaultGrouper();
            inputLength = Utility.sum(grouper.getClaimLayoutInputOnly());
            outputLength = Utility.sum(grouper.getClaimLayout()) - inputLength;
        } else {
//            LOG.log(Level.INFO, "Found Grouper for Date: {0}", fromDate);
            inputLength = Utility.sum(grouper.getClaimLayoutInputOnly());
            outputLength = Utility.sum(grouper.getClaimLayout()) - inputLength;
            retRes = grouper.group(Utility.adjustLine(claimInput, inputLength));
        }
        // generate input + output
        if (fullRecord) {
            return Utility.adjustLine(claimInput, inputLength)
                    + Utility.adjustLine(retRes, outputLength - 1) + "%";
        } else {
            return retRes;
        }
    }

    /**
     *
     * @param claim
     */
    protected void setResultForIncorrectDate(ClaimContainerIF claim) {
//            LOG.log(Level.INFO, "Grouper for Date: {0} was not found", fromDate);
        claim.setVersionUsed(getDefaultGrouper().getVersion());
        claim.setHippsCode("00000");
        // set validation code - new for v04.2.23 - Oct'23
        String vFlag = setValidityFlagForIncorrectDate(claim);
        ValidityFlagObject vfo = (ValidityFlagObject) 
                getDefaultGrouper().getAvailableTableContent(TableNames_EN.Validity_Flags).getDataMap().get(vFlag);
        ValidityFlagContainerIF vfc;
        if (vfo == null) {
            vfc = new ValidityFlagContainer("00", "Validity Flag not set");
        } else {
            vfc = new ValidityFlagContainer(vfo.getFlag(), vfo.getDescription());
        }
        claim.setValidityFlag(vfc);
        // set invalid date return code
        ReturnCodeObject rco = (ReturnCodeObject) getDefaultGrouper().getAvailableTableContent(TableNames_EN.Return_Codes).getDataMap().get("01");
        claim.setReturnCode(new ReturnCodeContainer(rco.getCode(), rco.getDescription()));
    }

    protected String setValidityFlagForIncorrectDate(ClaimContainerIF claim) {
        String EMPTY = "08";
        String INVALID = "09";
        String OUT_RANGE = "10";
        
        String dateFrom = claim.getFromDate();
        // 1. Empty ClaimFrom case
        if (dateFrom == null || dateFrom.trim().isEmpty()) {
            return EMPTY;
        }
        // 2. Not empty ClaimFrom case
        if ((new IsValidFromDate(dateFrom)).getDate() == null) {
            return INVALID;
        } else {
            return OUT_RANGE;
        }
    }

    /**
     *
     * @return
     */
    protected String getEmptyResult() {
        return "AA.A.AABBBBBCCDD";
    }

    /**
     *
     * @param claim
     * @return
     */
    protected String getResultForIncorrectDate(String claim) {
//            LOG.log(Level.INFO, "Grouper for Date: {0} was not found", fromDate);
        String claimInput = (claim == null? "" : claim);
        ClaimObject claimIn = getDefaultGrouper().getClaimManager().loadClaim(claimInput);
        setResultForIncorrectDate(claimIn);
        String retStr = claimIn.tempGetOutput();
        // Print logs
        String actionFlag = claimIn.getOptions().getActionFlag();
        if (actionFlag != null && actionFlag.equalsIgnoreCase(ActionFlag_EN.Y.name())) {
            LOG.info("*** Grouping Detail Report ***");
            LOG.info(claimIn.toString());
            LOG.log(Level.INFO, "*** Output string = {0}", retStr);
        }
        return retStr;
    }

    /**
     *
     * @return
     */
    @Override
    public GrouperIF getDefaultGrouper() {
        // look up for the default flag
        for (GrouperIF grouper : grouperList.values()) {
            if (grouper.isDefault()) {
                return grouper;
            }
        }
        return null;
    }

    private GrouperIF getGrouper(String claimDate) {
        // set claim date specs
        IsValidFromDate claimDateSpecs = new IsValidFromDate(claimDate);
        // look up for the appropriate grouper version
        for (GrouperIF grouper : grouperList.values()) {
            if (claimDateSpecs.isSatisfiedBy(grouper)) {
                return grouper;
            }
        }
        return null;
    }

    /**
     *
     * @param claimDate
     * @return
     */
    @Override
    public boolean isClaimDateValid(String claimDate) {
        boolean ret = false;
        if (getGrouper(claimDate) != null) {
            ret = true;
        }
        return ret;
    }

    private Map<String, GrouperIF> buildGrouperPool(String grouperVerFile, boolean loadDescription) {

        Map<String, GrouperIF> retGrouperList = null;

        InputStream inputFile = this.getClass().getClassLoader().getResourceAsStream(grouperVerFile);
        DataLoader<GrouperVersionObject> dl = new DataLoader(GrouperVersionObject.class, true); // load description
        List<GrouperVersionObject> grouperRtList = dl.loadFile(inputFile);

        // Create grouper pool/list
        try {
            for (int i = 0; i < grouperRtList.size(); i++) {
                // Skip not active versions
                if (Integer.parseInt(grouperRtList.get(i).getActiveFlag()) > 0) {
                    // Instantiate grouper
                    Class<?> clazz = Class.forName(grouperRtList.get(i).getJavaClass());
                    Constructor<?> constructor = clazz.getConstructor(Integer.class,
                            GrouperVersionObject.class, Boolean.class
                    );
                    Object instance = constructor.newInstance(i, grouperRtList.get(i), loadDescription);
                    // add to the list
                    GrouperIF grouper = (GrouperIF) instance;
                    if (retGrouperList == null) {
                        retGrouperList = new TreeMap();
                    }
                    retGrouperList.put(grouper.getName(), grouper);
                }
            }
        } catch (ClassNotFoundException
                | IllegalAccessException
                | IllegalArgumentException
                | InstantiationException
                | NoSuchMethodException
                | SecurityException
                | InvocationTargetException e) {
            Logger.getLogger(GrouperFactory.class
                    .getName()).log(Level.SEVERE, null, e);
        }

        // Set grouper next version links
        if (retGrouperList != null) {
            List<GrouperIF> grpList
                    = retGrouperList.values().stream().collect(Collectors.toList());
            // Set all links except last which can not have any link
            for (int i = 0; i < grpList.size() - 1; i++) {
                if (grpList.get(i).isLinked()) {
                    grpList.get(i).setNextVersion(grpList.get(i + 1));
                }
            }
            // Disable link flag for the latest version which can not have any link
            grpList.get(grpList.size() - 1).setUnlinked();
        }

        return retGrouperList;
    }

    /**
     *
     * @return
     */
    @Override
    public List<GrouperIF> getAvailableVersions() {
        return grouperList.values().stream().collect(Collectors.toList());
    }

    /**
     *
     * @return
     */
    @Override
    public String getBuildInfo() {

        String title = null;
        String version = null;
        Properties p = new Properties();
        InputStream res = null;

        // try to read directly from maven properties 
        try {
            res = getClass().getResourceAsStream("/META-INF/maven/gov.cms/hh/pom.properties");
            if (res != null) {
                p.load(res);
                version = p.getProperty("version");
                title = p.getProperty("artifactId");
                res.close();
            }
        } catch (IOException e) {
            // OK, do nothing
        }

        // If not try using Java API 
        if (version == null) {
            Package pack = getClass().getPackage();
            if (pack != null) {
                version = pack.getImplementationVersion();
                if (version == null) {
                    version = pack.getSpecificationVersion();
                }
                title = pack.getImplementationTitle();
                if (title == null) {
                    title = pack.getSpecificationTitle();
                }
            }
        }

        // OK, no luck
        if (version == null) {
            // Could not compute version
            version = "(unknown)";
        }

        if (title == null) {
            // Could not compute title
            title = "HH Grouper";
        }

        return "Software: " + title + ", Version: " + version;
    }

}
