package gov.cms.fiss.pricers.opps.core.tables;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.dataformat.csv.CsvParser.Feature;
import gov.cms.fiss.pricers.common.csv.CsvContentReader;
import gov.cms.fiss.pricers.common.csv.CsvIngestionConfiguration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class ProviderCcn340BLookupGenerator {

  private static final String CSV_FILE = "/providerCcn340B.csv";

  private final CsvContentReader<ProviderCcn340BEntry> contentProvider =
      new CsvContentReader<>(ProviderCcn340BEntry.class)
          .customizeSchema(schema -> schema.withHeader().withColumnReordering(true));

  public ProviderCcn340BLookupGenerator(CsvIngestionConfiguration csvIngestionConfiguration) {
    if (!csvIngestionConfiguration.isValidationEnabled()) {
      contentProvider.disableValidation();
    }

    contentProvider.customizeMapper(
        mapper ->
            (CsvMapper)
                mapper
                    .disable(Feature.FAIL_ON_MISSING_COLUMNS)
                    .disable(Feature.FAIL_ON_MISSING_HEADER_COLUMNS)
                    .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES));
  }

  /**
   * Creates a lookup set for Provider CCN 340B data from the default CSV file.
   *
   * @return the populated set of provider CCNs
   */
  public Set<String> generate() {
    return generate(CSV_FILE);
  }

  /**
   * Creates a lookup set for Provider CCN 340B data from the specified CSV file.
   *
   * @param csvFile filename of the csv file to use
   * @return the populated set of provider CCNs
   */
  public Set<String> generate(String csvFile) {
    final List<ProviderCcn340BEntry> content = contentProvider.read(csvFile);
    return content.stream()
        .map(ProviderCcn340BEntry::getProviderCcn)
        .collect(Collectors.toCollection(HashSet::new));
  }
}
