package gov.cms.grouper.snf.model;

import gov.cms.grouper.snf.lego.SnfUtils;

/**
 * Quality measures of the patients: Hearing, speech, vision, cognitive patterns, functional
 * abilities and goals, bladder and bowel...
 */
public class Assessment {

  public static final int NULL_VALUE = Integer.MIN_VALUE;
  public static final int CHECKED = 1;

  private final String name;
  private final String item; // xml tag
  private final int value;
  private final boolean isBlank;
  private final boolean isAssessed;

  /**
   * Constructs with a name, xml tag and value.
   * 
   * @param name non-null String.
   * @param item non-null XML name, can be the same as name.
   * @param value non-null double value.
   */
  public Assessment(String name, String item, Integer value) {
    this.name = name;
    this.item = item;
    this.value = value;
    this.isBlank = false;
    this.isAssessed = true;
  }

  /**
   * Constructs with a name, xml tag and value.
   *
   * @param name non-null String.
   * @param item non-null XML name, can be the same as name.
   * @param value non-null double value.
   */
  public Assessment(String name, String item, String value) {
    int tempVal;
    this.name = name;
    this.item = item;
    try {
      tempVal = Integer.parseInt(value.trim());
    } catch (NumberFormatException e) {
      tempVal = NULL_VALUE;
    }
    this.value = tempVal;
    this.isBlank = (value.isEmpty() || value.contains("^"));
    this.isAssessed = value.contains("-") == false;
  }

  public String getName() {
    return this.name;
  }

  public String getItem() {
    return this.item;
  }

  public int getValueInt() {
    return SnfUtils.nullCheck(this.value, Assessment.NULL_VALUE, this.value);
  }

  public boolean isBlank() {
    return this.isBlank;
  }

  public boolean isAssessed() {
    return this.isAssessed;
  }

  public boolean isCheck() {
    return this.getValueInt() == Assessment.CHECKED;
  }

  @Override
  public String toString() {
    return "Assessment{" + "name='" + this.name + '\'' + ", item='" + this.item + '\'' + ", value="
        + this.value + '}';
  }
}
