package gov.cms.grouper.snf.app;

import com.mmm.his.cer.foundation.exception.FoundationException;
import gov.cms.grouper.snf.process.SnfProcessor;
import gov.cms.grouper.snf.process.SnfValidations;
import gov.cms.grouper.snf.transfer.SnfClaim;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.function.Consumer;
import java.util.function.Function;

public class Pdpm {

  private static final Logger LOG = LoggerFactory.getLogger(Pdpm.class);
  private final SnfProcessorProxy processor;
  protected boolean ignoreExceptionOnTryExec;

  public Pdpm() {
    processor = new SnfProcessorProxy();
  }

  // PUBLIC METHODS

  /**
   * Process one MDS assessment through the PDPM grouper. Method is used in example C++ code for
   * calling the jar.
   *
   * @param line a fixed-length string in accordance with MDS 3.00.1 data specification
   * @return if the claim is a Proxy claim, then it returns the original record with the HIPPS code
   * placed within the record. Otherwise, it will just return the HIPPS Code (5 character string).
   */
  public static String stringExec(String line) {
    Pdpm pdpm = new Pdpm();
    return pdpm.handleLine(line, null).mapClaimToMds();
  }

  /**
   * Process one MDS assessment through the PDPM grouper.
   *
   * @param line a fixed-length string in accordance with MDS 3.00.1 data specification
   * @return the SnfClaim object with grouping results or {@code null} if encountered issues
   * Below are options to obtain the calculated results from the SnfClaim: - getHippsCode() : String
   * - getErrors() : List&lt;String&gt; - getVersion() : int
   */
  public SnfClaim exec(String line) {
    return handleLine(line, null);
  }

  /**
   * Process one MDS assessment through the PDPM grouper.
   *
   * @param claim a SnfClaim object with the necessary inputs set for the PDPM calculation
   * @return the SnfClaim object with grouping results or {@code null} if encountered issues
   * Below are options to obtain the calculated results from the SnfClaim: - getHippsCode() : String
   * - getErrors() : List&lt;String&gt; - getVersion() : int
   */
  public SnfClaim exec(SnfClaim claim) {
    return handleClaim(claim, null);
  }

  /**
   * Process one or more MDS assessment through the PDPM grouper.
   *
   * @param snfFile a file containing one or more fixed-length strings representing MDS assessments
   * @return a list of SnfClaim objects with grouping results or {@code null} if encountered
   * issues Below are options to obtain the calculated results from the SnfClaim: - getHippsCode() :
   * String - getErrors() : List&lt;String&gt; - getVersion() : int
   */
  public List<SnfClaim> exec(File snfFile) {
    List<SnfClaim> claims = new ArrayList<>(5000);
    exec(snfFile.toPath(), claims::add);
    return claims;
  }

  /**
   * This is used for Java JDK 32 processing of large data set. Because of memory limitations,
   * client will need to write out the result of SnfClaim into file or other storage in the
   * additionalPostProcessing.
   *
   * @param path                     file path
   * @param additionalPostProcessing processing after processing of claim
   */
  public void exec(Path path, Consumer<SnfClaim> additionalPostProcessing) {
    handleFile(path, (line) -> handleLine(line, additionalPostProcessing));

  }

  // PRIVATE METHODS

  private SnfClaim handleLine(String line, Consumer<SnfClaim> finalProcessing) {
    SnfClaim claim = SnfClaim.mapMdsToClaim(line);
    return handleClaim(claim, finalProcessing);
  }

  private SnfClaim handleClaim(SnfClaim claim, Consumer<SnfClaim> finalProcessing) {
    SnfValidations.validateInputs(claim);
    try {
      return processor.process(claim, finalProcessing);
    } catch (Throwable th) {
      if (ignoreExceptionOnTryExec) {
        finalProcessing.accept(claim);
        return claim;
      } else {
        throw new RuntimeException(th);
      }
    }

  }

  private static <T> void handleFile(Path path, Function<String, T> lineExec) {
    try (Scanner sc = new Scanner(path, "UTF-8")) {
      // close the file on JVM shutdown
      Thread closeFileThread = new Thread(() -> {
        LOG.debug("closing files");
        sc.close();
      });

      Runtime.getRuntime().addShutdownHook(closeFileThread);

      while (sc.hasNextLine()) {
        String line = sc.nextLine();
        if (sc.ioException() != null) {
          throw sc.ioException();
        }
        lineExec.apply(line);
      }
    } catch (Throwable th) {
      throw new RuntimeException(th);
    }
  }

  // PRIVATE CLASS PROXIES

  private static class SnfProcessorProxy extends SnfProcessor {
    @Override
    protected SnfClaim process(SnfClaim claim, Consumer<SnfClaim> finalProcessing)
        throws FoundationException, IOException {
      return super.process(claim, finalProcessing);
    }
  }
}
